/* Copyright (C) 2010 Monotype Imaging Inc. All rights reserved. */

/* Confidential information of Monotype Imaging Inc. */

/* fs_rtgah.c - RunTime Grayscale Autohinter */
/* works on data in the elementPtr */
/* developed by swp 5/14/10        */

/****************************************************/
/* Small ppm boost                                  */
/* undefine the following to disable the x and y    */
/* scale boost at small ppm sizes                   */
/****************************************************/
#define Y_BOOST /* define to enable artificial boost in x-heights at small ppm     */
#define X_BOOST /* define to make the x-height boost, also modify character widths */

/********************************************************************************/
/************************* F Y I ************************************************/
/********************************************************************************/
/* to add support for a new indic script xxx, with 2 ref_lines
 * 1) add the RTGA_xxx_top and RTGA_xxx_BOTTOM in object.h
 * 2) add the proper character fetching to FS_api.c, get_ref_lines()
 * 3) add the proper unicode range(s) to FS_glue.c, which_script()
 * 4) add the new snap_top_bottom() call to rtgsah.c, rtgah()
 */
/********************************************************************************/
#include "fs_api.h"
#ifdef FS_HINTS

#include "fs_rtgah.h"

/* define to get OODLES of debugging information
 * |= 1 for add_stroke
 * |= 2 for find_strokes
 * |= 4 for adjust_outline Y
 * |= 8 for stroke reduction
 * |= 16 for latin_reference_lines
 * |= 32 for find_pieces
 * |= 64 for multi-dot
 * |= 128 for my_iup(x)
 * |= 256 for my_iup(y)
 * |= 512 for latin_side_bearings
 * |= 2048 for latin_accents
 * |= 4096 non_corner_extrema
 * |= 8192 snap_top_bottom
 */
/* #define RTGAH_DEBUG (2+4+32+128+256) */

/* local functions -- alphabetical order */
static int     adjust_outline(char xy, int script, int num_strokes, STROKE *strokes, fnt_ElementType *elementPtr, fnt_LocalGraphicStateType *gs);
static int     can_move_stroke_down(int a, int b, int num_strokes, STROKE *strokes, F26DOT6 *coord, F26DOT6 *other);
static int     can_move_stroke_down_2(
#ifdef RTGAH_DEBUG
                                      int indent,
#endif
                                      int a, int b, int num_strokes, STROKE *strokes, F26DOT6 *coord, F26DOT6 *other);
static int     can_move_stroke_up(int a, int b, int num_strokes, STROKE *strokes, F26DOT6 *coord, F26DOT6 *other);
static int     can_move_stroke_up_2(
#ifdef RTGAH_DEBUG
                                    int indent,
#endif
                                    int a, int b, int num_strokes, STROKE *strokes, F26DOT6 *coord, F26DOT6 *other);
static int     can_thin_strokes(int a, int b, STROKE *strokes);
static int     find_pieces(char xy, PIECE *pieces, fnt_ElementType *elementPtr, PLIST **plist, PLIST *plistMax);
static int     find_pieces_CJK(char xy, PIECE *pieces, fnt_ElementType *elementPtr, PLIST **plist, PLIST *plistMax);
static int     find_strokes(char xy, int num_pieces, PIECE *pieces, STROKE *strokes, fnt_ElementType *elementPtr, F26DOT6 max_sw, PLIST **plist, PLIST *plistMax);
static int     find_strokes_CJK(char xy, int num_pieces, PIECE *pieces, STROKE *strokes, fnt_ElementType *elementPtr, F26DOT6 max_sw, PLIST **plist, PLIST *plistMax);
static int     is_contour_clockwise(int c, fnt_ElementType *elementPtr);
static int     latin_accents(F26DOT6 *ref_lines, int num_strokes, STROKE *strokes, fnt_ElementType *elementPtr, PLIST **plist, PLIST *plistMax, fnt_LocalGraphicStateType *gs);
static int     latin_ref_lines(int num_strokes, STROKE *strokes, F26DOT6 *ref_lines, fnt_ElementType *elementPtr, PLIST **plist, PLIST *plistMax);
static void    latin_side_bearings(int num_strokes, STROKE *strokes, F26DOT6 *ref_lines, fnt_ElementType *elementPtr);
static void    multi_dot(fnt_ElementType *elementPtr);
static void    no_collisions(char xy, int num_strokes, STROKE *strokes, fnt_ElementType *elementPtr);
static void    non_corner_extrema(fnt_ElementType *elementPtr);
static F26DOT6 pstroke_length(int p0, int p1, int p2, int p3, F26DOT6 *v);
static void    round_ref_lines(F26DOT6 *ref_lines, F26DOT6 *rrl);
static F26DOT6 simple_alignment(F26DOT6 center, F26DOT6 width);
static void    snap_top_bottom(F26DOT6 top, int num_strokes, STROKE *strokes);
static void    snap_top_bottom_4(F26DOT6 *rl, int num_strokes, STROKE *strokes, fnt_ElementType *elementPtr, F26DOT6 ppm);
static int     strokes_overlap(int a, int b, STROKE *strokes);
static int     strokes_too_close(int a, int b, STROKE *strokes);
static int     visually_separate(F26DOT6 a, F26DOT6 b);


/********************************************************************************/
#ifdef RTGAH_DEBUG /*************************************************************/
/********************************************************************************/
/* debugging glop is conditionally compiled */

static void dump_element(char *s, fnt_ElementType *elementPtr)
{
    FS_USHORT *ep = elementPtr->ep;
    FS_USHORT *sp = elementPtr->sp;
    FS_USHORT nc = elementPtr->nc;
    FS_USHORT c;
    FS_BYTE *f = elementPtr->f;
    F26DOT6 *x = elementPtr->x;
    F26DOT6 *ox = elementPtr->ox;
    F26DOT6 *y = elementPtr->y;
    F26DOT6 *oy = elementPtr->oy;
    double z = 64.0;
    int np, pt;

    if (nc == 0)
        return;
    FS_PRINTF(("dump_element --- %s\n", s));
    for (c = 0; c < nc; c++)
    {
        for (pt = sp[c]; pt <= ep[c]; pt++)
            FS_PRINTF(("%5d %02x %7.3f %7.3f %7.3f %7.3f \n", pt, f[pt], x[pt] / z, y[pt] / z, ox[pt] / z, oy[pt] / z));
        pt = sp[c];
        FS_PRINTF(("%5d %02x %7.3f %7.3f %7.3f %7.3f \n", pt, f[pt], x[pt] / z, y[pt] / z, ox[pt] / z, oy[pt] / z));
        FS_PRINTF(("\n"));
    }

    /* phantoms */
    np = 1 + ep[nc - 1];
    for (pt = np; pt < np + 4; pt++)
        FS_PRINTF(("%5d %02x %7.3f %7.3f %7.3f %7.3f \n", pt, f[pt], x[pt] / z, y[pt] / z, ox[pt] / z, oy[pt] / z));
    FS_PRINTF(("\n"));
}

/********************************************************************************/
static void dump_pieces(char *s, int num_pieces, PIECE *pieces, fnt_ElementType *elementPtr)
{
    FS_USHORT *ep = elementPtr->ep;
    FS_USHORT *sp = elementPtr->sp;
    F26DOT6 *x = elementPtr->x;
    F26DOT6 *ox = elementPtr->ox;
    F26DOT6 *y = elementPtr->y;
    F26DOT6 *oy = elementPtr->oy;
    int i, p0, p1;
    double z = 64.0;

    FS_PRINTF(("dump pieces --- %s\n", s));
    for (i = 0; i < num_pieces; i++)
    {
        PLIST *p;
        p0 = pieces[i].p0;
        p1 = pieces[i].p1;
        FS_PRINTF(("%5d %p %d %d %d %d %.3f %.3f | %.3f %.3f \n", i, pieces + i, pieces[i].c, pieces[i].d, p0, p1, x[p0] / z, y[p0] / z, ox[p0] / z, oy[p0] / z));

        FS_PRINTF(("     points %d %d ", p0, p1));
        for (p = pieces[i].pts; p; p = p->next)
            FS_PRINTF(("%d ", p->pt));
        FS_PRINTF(("\n"));
    }
}

/********************************************************************************/
/* in a way which makes graphing them easy */
/* vertical strokes */
static void dump_x_strokes(char *str, int n, STROKE *strokes, fnt_ElementType *elementPtr)
{
    int i;
    F26DOT6 x0, y0, x1, y1, x2, y2, x3, y3;
    F26DOT6 m, lo, hi;
    F26DOT6 *x, *y;
    double z = 64.0;

    FS_PRINTF(("dump_x_strokes -- %s\n", str));

    x = elementPtr->x;
    y = elementPtr->y;

    for (i = 0; i < n; i++)
    {
        x0 = x[strokes[i].p0];
        x1 = x[strokes[i].p1];
        x2 = x[strokes[i].p2];
        x3 = x[strokes[i].p3];

        y0 = y[strokes[i].p0];
        y1 = y[strokes[i].p1];
        y2 = y[strokes[i].p2];
        y3 = y[strokes[i].p3];

        if (y1 < y0)
        {
            m = y0;
            y0 = y1;
            y1 = m;
        }
        if (y3 < y2)
        {
            m = y2;
            y2 = y3;
            y3 = m;
        }

        m = strokes[i].adj_center;
        hi = m + strokes[i].width / 2;
        lo = m - strokes[i].width / 2;

        FS_PRINTF(("%f %f %d\n", lo / z, y0 / z, i));
        FS_PRINTF(("%f %f\n", lo / z, y1 / z));
        FS_PRINTF(("%f %f\n", hi / z, y3 / z));
        FS_PRINTF(("%f %f\n", hi / z, y2 / z));
        FS_PRINTF(("%f %f\n\n", lo / z, y0 / z));
    }
}

/********************************************************************************/
/* dump_strokes in a way which makes graphing them easy */
static void dump_y_strokes(char *str, int n, STROKE *strokes, fnt_ElementType *elementPtr)
{
    int i;
    F26DOT6 x0, y0, x1, y1, x2, y2, x3, y3;
    F26DOT6 m, lo, hi;
    F26DOT6 *x, *y;
    double z = 64.0;

    FS_PRINTF(("dump_y_strokes -- %s\n", str));

    x = elementPtr->x;
    y = elementPtr->y;

    for (i = 0; i < n; i++)
    {
        x0 = x[strokes[i].p0];
        x1 = x[strokes[i].p1];
        x2 = x[strokes[i].p2];
        x3 = x[strokes[i].p3];

        y0 = y[strokes[i].p0];
        y1 = y[strokes[i].p1];
        y2 = y[strokes[i].p2];
        y3 = y[strokes[i].p3];

        if (x1 < x0)    /* have the same y */
        {
            m = x0;
            x0 = x1;
            x1 = m;
        }
        if (x3 < x2)    /* have the same y */
        {
            m = x2;
            x2 = x3;
            x3 = m;
        }

        m = strokes[i].adj_center;
        hi = m + strokes[i].width / 2;
        lo = m - strokes[i].width / 2;

        FS_PRINTF(("%f %f %d\n", x0 / z, lo / z, i));
        FS_PRINTF(("%f %f\n", x1 / z, lo / z));
        FS_PRINTF(("%f %f\n", x3 / z, hi / z));
        FS_PRINTF(("%f %f\n", x2 / z, hi / z));
        FS_PRINTF(("%f %f\n\n", x0 / z, lo / z));
    }
}

/********************************************************************************/
static void dump_strokes(char *s, int num_strokes, STROKE *strokes)
{
    int i;
    double rc, ac, w, w2, z = 64.0;
    PLIST *p;

    FS_PRINTF(("dump_strokes --- %s\n", s));
    FS_PRINTF((" i p0 p1 p2 p3 w rc ac - lo hi\n"));
    for (i = 0; i < num_strokes; i++)
    {
        w = strokes[i].width;
        w2 = (1 + w) / 2;
        ac = strokes[i].adj_center;
        rc = strokes[i].raw_center;

        FS_PRINTF(("%5d %3d %3d %3d %3d %7.3f %7.3f %7.3f - %7.3f %7.3f\n", i, strokes[i].p0, strokes[i].p1,
                   strokes[i].p2, strokes[i].p3, w / z, rc / z, ac / z, (ac - w2) / z , (ac + w2) / z));

        if (strokes[i].e1)
        {
            FS_PRINTF(("    e1 "));
            for (p = strokes[i].e1; p; p = p->next)
                FS_PRINTF(("%d ", p->pt));
            FS_PRINTF(("\n"));
        }
        if (strokes[i].e2)
        {
            FS_PRINTF(("    e2 "));
            for (p = strokes[i].e2; p; p = p->next)
                FS_PRINTF(("%d ", p->pt));
            FS_PRINTF(("\n"));
        }
    }
    FS_PRINTF(("\n"));
}

/********************************************************************************/
static void dump_contours(char *s, int num, BOOJUM *p)
{
    int i;
    double z = 64.0;

    FS_PRINTF(("dump_contours --- %s\n", s));
    for (i = 0; i < num; i++)
    {
        FS_PRINTF(("%5d %5d %5d %5d %5d %7.3f %7.3f %7.3f %7.3f\n",
                   p[i].c, p[i].xmin, p[i].xmax, p[i].ymin, p[i].ymax,
                   p[i].oxc / z, p[i].oyc / z, p[i].xc / z, p[i].yc / z));
    }
}

/********************************************************************************/
#endif /* RTGAH_DEBUG ***********************************************************/
/********************************************************************************/

/* prepend <q> to <p> */
FS_INLINE static PLIST *add_to_plist(PLIST *p, PLIST *q)
{
    q->next = p;
    return q;
}

/* make a new PLIST element */
FS_INLINE static PLIST *new_plist(int pt, PLIST **plist)
{
    PLIST *p = *plist;
    p->pt = (FS_USHORT)pt;
    p->next = 0;
    (*plist)++;
    return p;
}

/* return a copy of the argument PLIST */
static PLIST *copy_plist(PLIST *p, PLIST **plist)
{
    PLIST *q = 0;

    for (/**/; p; p = p->next)
        q = add_to_plist(q, new_plist(p->pt, plist));

    return q;
}

/********************************************************************************/
/* stroke length wrt coordinate <v>,  */
/* length of middle of trapezoid formed by 2 edges (p0,p1) and (p2,p3) */
static F26DOT6 pstroke_length(int p0, int p1, int p2, int p3, F26DOT6 *v)
{
    F26DOT6  a, b, c, d, e;
    a = MIN(v[p0], v[p1]);
    b = MAX(v[p0], v[p1]);
    c = MIN(v[p2], v[p3]);
    d = MAX(v[p2], v[p3]);
    e = (b + d) - (a + c);
    if (e < 0)
        e = -e;
    return e / 2;
}

#define stroke_length(s,v) pstroke_length((s)->p0,(s)->p1,(s)->p2,(s)->p3,(v))

/********************************************************************************/
/* try and repair dumb characters
 * make "almost H/V" contour extrema, "actually H/V"
 * make short almost H/V segments, actually H/V
 *
 * unfortunately we can not **add** contour extrema to the character
 * so we can only fix things that are **close** to correct
 * so characters which contain no points near the extrema can't be helped
 */
static void non_corner_extrema(fnt_ElementType *elementPtr)
{
    FS_USHORT *ep = elementPtr->ep;
    FS_USHORT *sp = elementPtr->sp;
    FS_USHORT c, nc = elementPtr->nc;
    F26DOT6 *x = elementPtr->x;
    F26DOT6 *y = elementPtr->y;
    F26DOT6 *ox = elementPtr->ox;
    F26DOT6 *oy = elementPtr->oy;
    FS_BYTE *oc = elementPtr->onCurve;
    int j, xmax, xmin, ymax, ymin;
    int start, end;
    int p, n, np;
    F26DOT6 dx, dy;

#ifdef RTGAH_DEBUG
    if (RTGAH_DEBUG & 4096)
        dump_element("before non-corner-extrema", elementPtr);
#endif /* RTGAH_DEBUG */

    /* contour extrema */
    for (c = 0; c < nc; c++)
    {
        start = sp[c];
        end = ep[c];
        if (sp[c] == ep[c])
            continue;

        xmax = xmin = ymax = ymin = start;
        for (j = start + 1; j <= end; j++)
        {
            if (x[j] > x[xmax])
                xmax = j;
            else if (x[j] < x[xmin])
                xmin = j;

            if (y[j] > y[ymax])
                ymax = j;
            else if (y[j] < y[ymin])
                ymin = j;
        }

        p = PREV(xmax);
        n = NEXT(xmax);
        if (oc[xmax] == 0 || (oc[p] == 0 && oc[n] == 0))
        {
            if (x[p] != x[xmax] && x[xmax] != x[n])
            {
                dx = x[xmax] - x[p];
                if (dx < 0)
                    dx = -dx;
                dy = y[xmax] - y[p];
                if (dy < 0)
                    dy = -dy;
                if (ANG_TOL * dx < dy)
                    ox[p] = x[p] = x[xmax];

                dx = x[xmax] - x[n];
                if (dx < 0)
                    dx = -dx;
                dy = y[xmax] - y[n];
                if (dy < 0)
                    dy = -dy;
                if (ANG_TOL * dx < dy)
                    ox[n] = x[n] = x[xmax];
            }
        }

        p = PREV(xmin);
        n = NEXT(xmin);
        if (oc[xmin] == 0 || (oc[p] == 0 && oc[n] == 0))
        {
            if (x[p] != x[xmin] && x[xmin] != x[n])
            {
                dx = x[xmin] - x[p];
                if (dx < 0)
                    dx = -dx;
                dy = y[xmin] - y[p];
                if (dy < 0)
                    dy = -dy;
                if (ANG_TOL * dx < dy)
                    ox[p] = x[p] = x[xmin];

                dx = x[xmin] - x[n];
                if (dx < 0)
                    dx = -dx;
                dy = y[xmin] - y[n];
                if (dy < 0)
                    dy = -dy;
                if (ANG_TOL * dx < dy)
                    ox[n] = x[n] = x[xmin];
            }
        }

        p = PREV(ymax);
        n = NEXT(ymax);
        if (oc[ymax] == 0 || (oc[p] == 0 && oc[n] == 0))
        {
            if (y[p] != y[ymax] && y[ymax] != y[n])
            {
                dx = x[ymax] - x[p];
                if (dx < 0)
                    dx = -dx;
                dy = y[ymax] - y[p];
                if (dy < 0)
                    dy = -dy;
                if (ANG_TOL * dy < dx)
                    oy[p] = y[p] = y[ymax];

                dx = x[ymax] - x[n];
                if (dx < 0)
                    dx = -dx;
                dy = y[ymax] - y[n];
                if (dy < 0)
                    dy = -dy;
                if (ANG_TOL * dy < dx)
                    oy[n] = y[n] = y[ymax];
            }
        }

        p = PREV(ymin);
        n = NEXT(ymin);
        if (oc[ymin] == 0 || (oc[p] == 0 && oc[n] == 0))
        {
            if (y[p] != y[ymin] && y[ymin] != y[n])
            {
                dx = x[ymin] - x[p];
                if (dx < 0) dx = -dx;
                dy = y[ymin] - y[p];
                if (dy < 0) dy = -dy;
                if (ANG_TOL * dy < dx)
                    oy[p] = y[p] = y[ymin];

                dx = x[ymin] - x[n];
                if (dx < 0) dx = -dx;
                dy = y[ymin] - y[n];
                if (dy < 0) dy = -dy;
                if (ANG_TOL * dy < dx)
                    oy[n] = y[n] = y[ymin];
            }
        }
    }

    np = 1 + ep[nc - 1];
    SYS_MEMCPY(x, ox, np * 4);
    SYS_MEMCPY(y, oy, np * 4);

#ifdef RTGAH_DEBUG
    if (RTGAH_DEBUG & 4096)
        dump_element("after non-corner-extrema", elementPtr);
#endif /* RTGAH_DEBUG */
}

/*lint -e771 Info -- Symbol conceivably not initialized */
/********************************************************************************/
static int latin_accents(F26DOT6 *ref_lines, int num_strokes, STROKE *strokes, fnt_ElementType *elementPtr,
                         PLIST **pplist, PLIST *plistMax, fnt_LocalGraphicStateType *gs)
{
    F26DOT6 *x = elementPtr->x;
    F26DOT6 *y = elementPtr->y;
    F26DOT6 *ox = elementPtr->ox;
    F26DOT6 *oy = elementPtr->oy;
    FS_BYTE *f = elementPtr->f;
    FS_USHORT *ep = elementPtr->ep;
    FS_USHORT *sp = elementPtr->sp;
    FS_USHORT nc = elementPtr->nc;
    FS_USHORT i, j;
    int any;
    int *ymin = (int *)gs->globalGS->rtgah_data.RTGAHYmin;
    int *ymax = (int *)gs->globalGS->rtgah_data.RTGAHYmax;
    int *xmin = (int *)gs->globalGS->rtgah_data.RTGAHXmin;
    int *xmax = (int *)gs->globalGS->rtgah_data.RTGAHXmax;

    char outside[MAX_CONTOURS];
    F26DOT6 z, zz, h, new_h;
    FS_FIXED s;
    int base_contour;
    F26DOT6 old_top_of_base, new_top_of_base;
    F26DOT6 old_ymin, new_ymin, old_ymax, dy;

    if (nc > MAX_CONTOURS)
        nc = MAX_CONTOURS;

#ifdef RTGAH_DEBUG
    if (RTGAH_DEBUG & 2048)
        dump_element("before Latin accents", elementPtr);
#endif /* RTGAH_DEBUG */

    /* contour extrema */
    for (i = 0; i < nc; i++)
    {
        j = sp[i];
        xmin[i] = xmax[i] = j;
        ymin[i] = ymax[i] = j;
        for (j++; j <= ep[i]; j++)
        {
            if (y[j] > y[ymax[i]])
                ymax[i] = j;
            else if (y[j] < y[ymin[i]])
                ymin[i] = j;
            if (x[j] > x[xmax[i]])
                xmax[i] = j;
            else if (x[j] < x[xmin[i]])
                xmin[i] = j;
        }
    }

    /* find which contour has the left-most point.
    find the direction of that contour -- note, it is outside.
    then set all contours of that direction to outside. */
    {
        F26DOT6 lo = x[xmin[0]];
        int c = 0;
        int cw;

        for (i = 1; i < nc; i++)
        {
            if (x[xmin[i]] < lo)
            {
                lo = x[xmin[i]];
                c = i;
            }
        }

        /* direction of the leftmost contour */
        cw = is_contour_clockwise(c, elementPtr);

        for (i = 0; i < nc; i++)
        {
            c = is_contour_clockwise(i, elementPtr);
            outside[i] = (c == cw);
        }
    }

    /* top of the tallest contour,  */
    /* this **should** be the top of the base character */
    base_contour = 0;   /* assure the compiler this it is always set */
    z = 0;
    for (i = 0; i < nc; i++)
    {
        zz = y[ymax[i]] - y[ymin[i]];
        if (zz > z)
        {
            base_contour = i;
            z = zz;
        }
    }
    old_top_of_base = new_top_of_base = y[ymax[base_contour]];

    /* Top of base character should be above x_height */
    /* Well ... not more than 1/2 pixel lower */
    if (old_top_of_base < (ref_lines[RTGA_LC_SQUARE] - 32))
        return num_strokes;

    /* Bottom of base character should be the baseline */
    /* Well ... not more than 1/2 pixel above it */
    if (y[ymin[base_contour]] > (ref_lines[RTGA_BASE_SQUARE] + 32))
        return num_strokes;

    /* don't consider single point contours */
    for (i = 0; i < nc; i++)
    {
        if (sp[i] == ep[i])
            outside[i] = 0;
    }

    /* don't adjust any outside contours which have an inside contour -- Ring accent issues */
    /* do this by clearing the 'outside' bit of such a contour  */
    for (i = 0; i < nc; i++)
    {
        if (outside[i])
        {
            for (j = 0; j < nc; j++)
            {
                if (j == i)
                    continue;
                if (y[ymin[j]] > y[ymin[i]] &&  y[ymax[j]] < y[ymax[i]] &&
                        x[xmin[j]] > x[xmin[i]] &&  x[xmax[j]] < x[xmax[i]])
                {
                    outside[i] = 0;
                    break;
                }
            }
        }
    }

    /* don't adjust any contours which are not entirely above the base_contour */
    /* do this by clearing the bit of such a contour  */
    /* outside[] will then mark the accent contours we will play with */
    /* important that we are using the pre-alignment value for top_of_base */
    for (any = 0, i = 0; i < nc; i++)
    {
        if (y[ymin[i]] < old_top_of_base)
            outside[i] = 0;
        any |= outside[i];
    }

    /* no suitable contours entirely above the base_contour ... bail */
    if (!any)
        return num_strokes;

    /* determine where top_of_base will end up after strokes are aligned */
    for (i = 0; i < num_strokes; i++)
    {
        F26DOT6 c, w, hi;

        c = strokes[i].raw_center;
        w = strokes[i].width;
        hi = c + w / 2;

        if (ABS(hi - old_top_of_base) < 16)
        {
            /* it's in a stroke, it will be aligned thusly */
            new_top_of_base = (w / 2) + strokes[i].adj_center;
            break;
        }
    }


    /* top_of_base not in a stroke ... add one */
    if (i == num_strokes)
    {
        STROKE *st = strokes + num_strokes++;
        int np = 1 + ep[nc - 1];
        st->p0 = (FS_USHORT)(ymax[base_contour]);
        st->p1 = st->p2 = st->p3 = st->p0;
        st->width = 0;
        st->raw_center = old_top_of_base;
        st->e1 = st->e2 = 0;
        st->max = x[xmax[base_contour]];
        st->min = x[xmin[base_contour]];

        /* add other points at old_top_of_base */
        for (i = 0, j = st->p0; i < np; i++)
        {
            if (i != j && old_top_of_base == y[i])
            {
                st->e1 = add_to_plist(st->e1, new_plist(i, pplist));
                if (*pplist >= plistMax)
                    break;
            }
        }


        /* it will end up here */
        new_top_of_base = ROUND_26_6(old_top_of_base);
        st->adj_center = new_top_of_base;
    }

    /* find old_ymin,old_ymax for accent contours */
    old_ymin = MYINFINITY;
    old_ymax = -MYINFINITY;
    for (i = 0; i < nc; i++)
    {
        if (!outside[i])
            continue;
        if (y[ymin[i]] < old_ymin)
            old_ymin = y[ymin[i]];
        if (y[ymax[i]] > old_ymax)
            old_ymax = y[ymax[i]];
    }

    dy = old_ymin - old_top_of_base;
    dy = ROUND_26_6(dy);
    if (dy == 0)
        dy = 64;
    new_ymin = new_top_of_base + dy;

    /* it would be nice if we could tell the difference here, between
     * accents which require a minimum height of 2 pixels (tilde, acute,
     * breve, caron, circumflex...) and the ones which can live at a
     * height of 1 pixel (dot, macron, diaeresis)
     *
     * we only *need* to boost the height of the former set. but today
     * we boost the height of all accents to be at least the minimum size.
     */
    h = old_ymax - old_ymin;
    if (h == 0)                 /* could be a degenerate contour ?? */
        return num_strokes;

    if (h > 64 && h <= 128)
        new_h = 128;
    else
        new_h = h;

    s = (new_h << 16) / h;
    for (i = 0; i < nc; i++)
    {
        if (!outside[i])
            continue;

        for (j = sp[i]; j <= ep[i]; j++)
        {
            y[j] = new_ymin + FixMul(s, oy[j] - old_ymin);
            f[j] |= YMOVED;
        }
    }

    /* if small, center each accent contour wrt x, in a pixel */
    if (nc > 2) /* exclude i and j which *probably* have 2 contours */
    {
        for (i = 0; i < nc; i++)
        {
            F26DOT6 m, new_m, dx;

            if (!outside[i])
                continue;
            if (XMOVED & (f[xmax[i]] | f[xmin[i]]))
                continue;

            if (x[xmax[i]] - x[xmin[i]] < 128)
            {
                m = (x[xmax[i]] + x[xmin[i]]) / 2;
                new_m = 32 + FLOOR_26_6(m);
                dx = new_m - m;
                for (j = sp[i]; j <= ep[i]; j++)
                {
                    x[j] += dx;
                }
            }
        }
    }

    /* revise the centers/widths of the strokes ABOVE the top_of_base */
    for (i = 0; i < num_strokes; i++)
    {
        STROKE *si;
        F26DOT6 p0, p2;

        si = strokes + i;
        p0 = y[si->p0];
        p2 = y[si->p2];
        if (si->raw_center > old_top_of_base)
        {
            si->adj_center = si->raw_center = (p0 + p2) / 2;
            si->width = ABS(p2 - p0);
        }
    }

#ifdef RTGAH_DEBUG
    if (RTGAH_DEBUG & 2048)
    {
        dump_element("after Latin accents", elementPtr);
        dump_strokes("after Latin accents", num_strokes, strokes);
    }
#endif /* RTGAH_DEBUG */

    /* copy x[] to ox[], and y[] to oy[] */
    /* since that's where we look in the rest of hinting */
    j = 1 + ep[nc - 1];
    j <<= 2;
    SYS_MEMCPY(ox, x, j);
    SYS_MEMCPY(oy, y, j);

    return num_strokes;
}


/********************************************************************************/
/* impose l's side-bearings at (0,1) or (1,1) spacings of the "l" */
/* above that, it lets the default stroke alignment do it's work. */
static void latin_side_bearings(int num_strokes, STROKE *strokes, F26DOT6 *ref_lines, fnt_ElementType *elementPtr)
{
    F26DOT6 *x = elementPtr->x;
    F26DOT6 *ox = elementPtr->ox;
    FS_USHORT *ep = elementPtr->ep;
    FS_USHORT nc = elementPtr->nc;
    FS_USHORT i, n, np = 1 + ep[nc - 1];
    F26DOT6 max, min;
    F26DOT6 l_lsb, l_rsb, l_w;
    F26DOT6 lsb, rsb;
    F26DOT6 c0, w0, cn, wn;
    F26DOT6 L, R, dL, dR, new_L, new_R;
    F26DOT6 z, aw;

    /* dump_element("entering latin_side_bearings",elementPtr); */

    if (num_strokes == 0)
        return;
    l_lsb = ref_lines[RTGA_l_LSB];
    l_rsb = ref_lines[RTGA_l_RSB];
    l_w = l_lsb + l_rsb;

    /* ? l's side bearings > (1,1), then use the default stroke alignment */
    if (l_w > 128)
        return;

    /* advance width */
    aw = x[1 + np];

    /* char extrema */
    max = min = x[0];
    for (i = 1; i < np; i++)
    {
        if (x[i] > max)
            max = x[i];
        if (x[i] < min)
            min = x[i];
    }

    /* character's left and right stem edges */
    n = (FS_USHORT)(num_strokes - 1);
    c0 = strokes[0].raw_center;
    cn = strokes[n].raw_center;
    w0 = strokes[0].width / 2;
    wn = strokes[n].width / 2;
    L = c0 - w0;
    R = cn + wn;

    /* distance from outside stems to edges of ink */
    lsb = L;
    rsb = aw - R;

    /* ? one effective stem: c f i j k l r t ... maybe e */
    if (ABS(c0 - cn) < 32)
    {
        F26DOT6 dx = 0;

        /* distance to edges of ink */
        R = c0 + w0;
        dL = L - min;
        dR = max - R;

        /* ? stroke on left side of the ink: c f k r */
        if (dL <= dR && dL < 64)
        {
            if (ABS(l_lsb - lsb) <= 64)
            {
                /* use l_lsb */
                new_L = l_lsb;
                dx = new_L - L;
            }
        }

        /* ? stroke on right side of the ink: j */
        if (dR <= dL && dR < 64)
        {
            if (ABS(l_rsb - rsb) <= 64)
            {
                /* use l_rsb */
                new_R = aw - l_rsb;
                dx = new_R - R;
            }
        }

        if (dx)
        {
            for (i = 0; i < np; i++)
                ox[i] = x[i] += dx;
            for (i = 0; i < num_strokes; i++)
            {
                z = strokes[i].raw_center + dx;
                strokes[i].adj_center = z;
                strokes[i].raw_center = z;
            }
        }
    }
    else
        /* multiple stems: a b d g h m n o p q s u ... maybe e */
    {
        /* ? close to left edge and close to l_lsb  */
        dL = L - min;
        new_L = L;
        if (dL < 64)
        {
            if (ABS(l_lsb - lsb) <= 72)     /* 72==9/8 pixels is an empirical constant */
                new_L = l_lsb;              /* in other words ... it seems to work !!! */
        }

        /* ? close to right edge and close to l_rsb  */
        dR = max - R;
        new_R = R;
        if (dR < 64)
        {
            if (ABS(l_rsb - rsb) <= 72)
                new_R = aw - l_rsb;
        }

        /* interpolate between new edges */
        if (new_L != L || new_R != R)
        {
            FS_FIXED sx = 4194304L; /* one in 10.22 */
            F26DOT6 dx;

            /* sx is really 10.22 ... but it's ok.  the subsequent */
            /* FixMul removes 16 fractional bits from the product */
            /* and gets us back to a 26.6 ... Really it's OK. */
            if (R != L)
                sx = FixDiv(new_R - new_L, R - L);
            dx = new_L - L;

            for (i = 0; i < np; i++)
            {
                if (x[i] < L)
                    x[i] += dx;
                else if (x[i] <= R)
                    x[i] = new_L + FixMul(sx, x[i] - L);
                else
                    x[i] += new_R - R;
                ox[i] = x[i];
            }

            for (i = 0; i < num_strokes; i++)
            {
                STROKE *s;

                s = strokes + i;
                z = (x[s->p0] + x[s->p2]) / 2;
                s->raw_center = s->adj_center = z;
            }
        }
    }
    /* dump_element("leaving latin_side_bearings",elementPtr); */
}

/********************************************************************************/
static void round_ref_lines(F26DOT6 *ref_lines, F26DOT6 *rrl)
{
    F26DOT6 p0, p2;
    int i;

    for (i = 0; i < RTGA_COUNT; i++)
        rrl[i] = ROUND_26_6(ref_lines[i]);

    /* ? what are the round/square differences */
    p0 = ref_lines[RTGA_CAP_ROUND] - ref_lines[RTGA_CAP_SQUARE];
    p2 = ref_lines[RTGA_BASE_SQUARE] - ref_lines[RTGA_BASE_ROUND];

    /* ? eliminate the round/square differences */
    /* note: 44 makes typical fonts break near 50 ppm as desired */
    if (p0 < 44 || p2 < 44)
    {
        rrl[RTGA_CAP_ROUND] = rrl[RTGA_CAP_SQUARE];
        /* give the x-height a boost, by using LC_ROUND */
        rrl[RTGA_LC_SQUARE] = rrl[RTGA_LC_ROUND];
        rrl[RTGA_BASE_ROUND] = rrl[RTGA_BASE_SQUARE];
    }
}

/********************************************************************************/
#define LOCAL_MAX (1)
#define LOCAL_MIN (-1)
static int local_extrema(int i, fnt_ElementType *elementPtr)
{
    int p, n;
    F26DOT6 *oy = elementPtr->oy;
    FS_USHORT *sp = elementPtr->sp;
    FS_USHORT *ep = elementPtr->ep;
    FS_USHORT c, nc = elementPtr->nc;
    register F26DOT6 oy_i = oy[i];

    /* ? what contour we in */
    for (c = 0; c < nc; c++)
    {
        if (sp[c] <= i && i <= ep[c])
            break;
    }
    if (c == nc)
        return 0;

    /* ? single point contour */
    if (sp[c] == ep[c])
        return 0;

    /* ? contour has constant y */
    for (p = sp[c]; p <= ep[c]; p++)
    {
        if (oy[p] != oy[sp[c]])
            break;
    }
    if (p > ep[c])
        return 0;

    p = PREV(i);
    while (oy_i == oy[p])
        p = PREV(p);

    n = NEXT(i);
    while (oy_i == oy[n])
        n = NEXT(n);

    if (oy[p] > oy_i && oy[n] > oy_i)
        return LOCAL_MIN;
    if (oy[p] < oy_i && oy[n] < oy_i)
        return LOCAL_MAX;
    return 0;   /* neither */
}

/********************************************************************************/
/* latin Round/Square alignments */
static int latin_ref_lines(int num_strokes, STROKE *strokes, F26DOT6 *ref_lines, fnt_ElementType *elementPtr,
                           PLIST **pplist, PLIST *plistMax)
{
    F26DOT6 rrl[RTGA_COUNT];
    F26DOT6 c, w, hi, lo;
    int i;
    STROKE *s;
    F26DOT6 *oy = elementPtr->oy;
    FS_USHORT *ep = elementPtr->ep;
    F26DOT6 *other = elementPtr->ox;
    int nc = elementPtr->nc;
    int np = 1 + ep[nc - 1];
    int min, max, x_height, base;
    register F26DOT6 z;
    F26DOT6 ymin, ymax; /* in this cse it is actually x-coor */

    round_ref_lines(ref_lines, rrl);

#if 1
    /* pts where y hits extrema, x-height and baseline */
    max = min = 0;
    x_height = base = -1;
    ymin = ymax = other[0];
    for (i = 1; i < np; i++)
    {
        z = other[i];
        if (z < ymin)
        {
            ymin = z;
        }
        else if (z > ymax)
        {
            ymax = z;
        }

        if (oy[i] > oy[max])
            max = i;
        if (oy[i] < oy[min])
            min = i;
        if ((ABS(oy[i] - ref_lines[RTGA_LC_SQUARE]) <= 3) && LOCAL_MAX == local_extrema(i, elementPtr))
            x_height = i;
        else if ((ABS(oy[i] - ref_lines[RTGA_BASE_SQUARE]) <= 3) && LOCAL_MIN == local_extrema(i, elementPtr))
            base = i;
    }
#else
    /* pts where y hits extrema, x-height and baseline */
    max = min = 0;
    x_height = base = -1;
    for (i = 1; i < np; i++)
    {
        if (oy[i] > oy[max])
            max = i;
        if (oy[i] < oy[min])
            min = i;
        if (ABS(oy[i] - ref_lines[RTGA_LC_SQUARE]) < 3)
            x_height = i;
        else if (ABS(oy[i] - ref_lines[RTGA_BASE_SQUARE]) < 3)
            base = i;
    }
#endif

    /* ? max handled by a stroke */
    for (i = num_strokes - 1; i >= 0; i--)
    {
        c = strokes[i].raw_center;
        w = strokes[i].width / 2;
        lo = (c - w) - 1;
        hi = (c + w) + 1;
        if (lo <= oy[max] && oy[max] <= hi)
            break;
    }
    if (i < 0)
    {
        /* no ... add one */
        s = strokes + num_strokes++;
        s->p0 = (FS_USHORT)max;
        s->p1 = (FS_USHORT)max;
        s->p2 = (FS_USHORT)max;
        s->p3 = (FS_USHORT)max;
        s->raw_center = oy[max];
        s->adj_center = oy[max];
        s->width = 0;
        s->e1 = s->e2 = 0;
        s->max = ymax;
        s->min = ymin;

        /* add other points with same y coord */
        for (i = 0; i < np; i++)
        {
            if (i != max && oy[i] == oy[max])
            {
                s->e1 = add_to_plist(s->e1, new_plist(i, pplist));
                if (*pplist >= plistMax)
                    break;
            }
        }
    }


    /* ? min handled by a stroke */
    for (i = 0; i < num_strokes; i++)
    {
        c = strokes[i].raw_center;
        w = strokes[i].width / 2;
        lo = (c - w) - 1;
        hi = (c + w) + 1;
        if (lo <= oy[min] && oy[min] <= hi)
            break;
    }
    if (i == num_strokes)
    {
        /* no ... add one */
        s = strokes + num_strokes++;
        s->p0 = (FS_USHORT)min;
        s->p1 = (FS_USHORT)min;
        s->p2 = (FS_USHORT)min;
        s->p3 = (FS_USHORT)min;
        s->raw_center = oy[min];
        s->adj_center = oy[min];
        s->width = 0;
        s->e1 = s->e2 = 0;
        s->max = ymax;
        s->min = ymin;

        /* add other points with same y coord */
        for (i = 0; i < np; i++)
        {
            if (i != min && oy[i] == oy[min])
            {
                s->e1 = add_to_plist(s->e1, new_plist(i, pplist));
                if (*pplist >= plistMax)
                    break;
            }
        }

    }

    /* ? x_height present */
    if (x_height >= 0)
    {
        /* ? x_height handled by a stroke  */
        for (i = 0; i < num_strokes; i++)
        {
            c = strokes[i].raw_center;
            w = strokes[i].width / 2;
            lo = (c - w) - 1;
            hi = (c + w) + 1;
            if (lo <= oy[x_height] && oy[x_height] <= hi)
                break;
        }
        if (i == num_strokes)
        {
            /* no ... add one */
            s = strokes + num_strokes++;
            s->p0 = (FS_USHORT)x_height;
            s->p1 = (FS_USHORT)x_height;
            s->p2 = (FS_USHORT)x_height;
            s->p3 = (FS_USHORT)x_height;
            s->raw_center = oy[x_height];
            s->adj_center = oy[x_height];
            s->width = 0;
            s->e1 = s->e2 = 0;
            s->max = ymax;
            s->min = ymin;

            /* add other points with same y coord */
            for (i = 0; i < np; i++)
            {
                if (i != x_height && oy[i] == oy[x_height])
                {
                    s->e1 = add_to_plist(s->e1, new_plist(i, pplist));
                    if (*pplist >= plistMax)
                        break;
                }
            }
        }
    }

    /* ? base present, and different from the min ??? */
    if (base >= 0  && oy[base] != oy[min])
    {
        s = strokes + num_strokes++;
        s->p0 = (FS_USHORT)base;
        s->p1 = (FS_USHORT)base;
        s->p2 = (FS_USHORT)base;
        s->p3 = (FS_USHORT)base;
        s->raw_center = oy[base];
        s->adj_center = oy[base];
        s->width = 0;
        s->e1 = s->e2 = 0;
        s->max = ymax;
        s->min = ymin;
        /* add other points with same y coord */
        for (i = 0; i < np; i++)
        {
            if (i != base && oy[i] == oy[base])
            {
                s->e1 = add_to_plist(s->e1, new_plist(i, pplist));
                if (*pplist >= plistMax)
                    break;
            }
        }
    }

    /* snap strokes to reference lines */
    for (i = 0; i < num_strokes; i++)
    {
        c = strokes[i].raw_center;
        w = strokes[i].width / 2;
        lo = c - w;
        hi = c + w;

        /* snap to square first, then round. Promotes uniformity at small sizes */
        if (ABS(hi - ref_lines[RTGA_CAP_SQUARE]) < CAP_REF_LINE_TOL)
            strokes[i].adj_center = rrl[RTGA_CAP_SQUARE] - w;
        else if (ABS(hi - ref_lines[RTGA_CAP_ROUND]) < CAP_REF_LINE_TOL)
            strokes[i].adj_center = rrl[RTGA_CAP_ROUND] - w;
        else if (ABS(hi - ref_lines[RTGA_LC_SQUARE]) < LC_REF_LINE_TOL)
            strokes[i].adj_center = rrl[RTGA_LC_SQUARE] - w;
        else if (ABS(hi - ref_lines[RTGA_LC_ROUND]) < LC_REF_LINE_TOL)
            strokes[i].adj_center = rrl[RTGA_LC_ROUND] - w;
        else if (ABS(lo - ref_lines[RTGA_BASE_SQUARE]) < BASE_REF_LINE_TOL)
            strokes[i].adj_center = rrl[RTGA_BASE_SQUARE] + w;
        else if (ABS(lo - ref_lines[RTGA_LC_ROUND]) < BASE_REF_LINE_TOL)
            strokes[i].adj_center = rrl[RTGA_LC_ROUND] + w;
    }

#ifdef RTGAH_DEBUG
    if (RTGAH_DEBUG & 16)
    {
        dump_strokes("after latin_ref_lines", num_strokes, strokes);
    }
#endif  /* RTGAH_DEBUG */

    return num_strokes;
}

/********************************************************************************/
/* adjust the tops of strokes which are *near* <top> to be *at* <top> */
static void snap_top_bottom(F26DOT6 top, int num_strokes, STROKE *strokes)
{
    int i;
    STROKE *si;
    F26DOT6 w, hi;
    F26DOT6 rtop = ROUND_26_6(top);

    for (i = 0; i < num_strokes; i++)
    {
        si = strokes + i;
        w = si->width / 2;
        hi = si->raw_center + w;
        if (ABS(hi - top) < 64)
            si->adj_center = rtop - w;
    }
}

/********************************************************************************/
/* ref_lines for Kannada, Tamil, Malayalam...
*  rl[0] = cap_round
*  rl[1] = cap_square
*  rl[2] = base_square
*  rl[3] = base_round
*/
static void Indic_ref_lines_4(F26DOT6 *rl, int num_strokes, STROKE *strokes)
{
    F26DOT6 rrl[4], top, bot, w;
    int i;

    /* round the squares */
    rrl[1] = ROUND_26_6(rl[1]);
    rrl[2] = ROUND_26_6(rl[2]);

    /* add rounded overshoots */
    w = rl[0] - rl[1];  /* cap overshoot */
    rrl[0] = rrl[1] + ROUND_26_6(w);
    w = rl[2] - rl[3];  /* base overshoot */
    rrl[3] = rl[2] - ROUND_26_6(w);

    /* snap if close */
    for (i = 0; i < num_strokes; i++)
    {
        w = strokes[i].width / 2;
        top = strokes[i].raw_center + w;
        if (ABS(rl[1] - top) < 32)
            strokes[i].adj_center = rrl[1] - w;    /* cap_square */
        else if (ABS(rl[0] - top) < 32)
            strokes[i].adj_center = rrl[0] - w;    /* then cap_round */

        bot = strokes[i].raw_center - w;
        if (ABS(rl[2] - bot) < 32)
            strokes[i].adj_center = rrl[2] + w;    /* base_square */
        else if (ABS(rl[3] - bot) < 32)
            strokes[i].adj_center = rrl[3] + w;    /* then base_round */
    }
}
/********************************************************************************/
/* rl[0] = ymax */
/* rl[1] = ymin */
static void Telugu_ref_lines(F26DOT6 *rl, int num_strokes, STROKE *strokes)
{
    F26DOT6 rrl[2], top, bot, w;
    int i;

    rrl[0] = ROUND_26_6(rl[0]);
    rrl[1] = ROUND_26_6(rl[1]);

    for (i = 0; i < num_strokes; i++)
    {
        w = strokes[i].width / 2;
        top = strokes[i].raw_center + w;
        if (ABS(top - rl[0]) < 32)
            strokes[i].adj_center = rrl[0] - w;

        bot = strokes[i].raw_center - w;
        if (ABS(bot - rl[1]) < 32)
            strokes[i].adj_center = rrl[1] + w;
    }
}

/********************************************************************************/
/* rl[0] = ymin */
static void Arabic_ref_lines(F26DOT6 *rl, int num_strokes, STROKE *strokes)
{
    F26DOT6 rrl[1], c, bot, w;
    int i;

    rrl[0] = ROUND_26_6(rl[0]);

    for (i = 0; i < num_strokes; i++)
    {
        w = strokes[i].width / 2;
        c = strokes[i].raw_center;
        bot = c - w;
        if (ABS(bot - rl[0]) < 32)
            strokes[i].adj_center = rrl[0] + w;
    }
}
/********************************************************************************/
/* USE A PAIR OF CONTROL CHARACTERS FOR FLAT AND POINTY CHARACTERS */
static void snap_top_bottom_4(F26DOT6 *rl, int num_strokes, STROKE *strokes, fnt_ElementType *elementPtr, F26DOT6 ppm)
{
    F26DOT6 *x = elementPtr->x;
    F26DOT6 *y = elementPtr->y;
    F26DOT6 *oy = elementPtr->oy;
    FS_USHORT *ep = elementPtr->ep;
    FS_USHORT nc = elementPtr->nc;
    F26DOT6 min, max, new_max, new_min;
    F26DOT6 hi, lo;
    F26DOT6 rrl[4];
    int i, n, np;
    STROKE *si;
    F26DOT6 p0, p2;
    F26DOT6 len_lo, len_hi, min_len = ppm / 6; /* takes a pretty long stroke to be considered a 'flat top' or 'flat bottom' */

    if (nc == 0)
        return;
    np = 1 + ep[nc - 1];

    /*** round reference lines ***/
    /* round pointy bot */
    rrl[3] = ROUND_26_6(rl[3]);

    /* what is that '6+' doing in the next line of code ??? */
    /* an empiracle constant ... */
    /* infer pointy top */
    rrl[0] = rrl[3] + ROUND_26_6(6 + rl[0] - rl[3]);

    /* infer flat bottom from pointy bottom */
    /* insure one pixel difference above 14 ppm*/
    p0 = ROUND_26_6(rl[2] - rl[3]);
    if (ppm > 896 && p0 <= 0)
        p0 = 64;
    rrl[2] = rrl[3] + p0;

    /* infer flat top from pointy top */
    /* insure one pixel difference above 14ppm*/
    p0 = ROUND_26_6(rl[0] - rl[1]);
    if ( ppm  > 896 && p0 <= 0)
        p0 = 64;
    rrl[1] = rrl[0] - p0;

    /* glyph extrema */
    min = max = y[0];
    for (i = 1; i < np; i++)
    {
        if (y[i] < min)
            min = y[i];
        if (y[i] > max)
            max = y[i];
    }
    new_max = max;
    new_min = min;
    if (max == min)
        return;

    if (num_strokes == 0)
    {
        if (ABS(max - rl[0]) < CJK_TROUGH)
            new_max = rrl[0];
        if (ABS(min - rl[3]) < CJK_TROUGH)
            new_min = rrl[3];
    }
    else
    {
        /* bottom of lowest stroke, top of highest stroke */
        lo = strokes[0].raw_center - strokes[0].width / 2;
        n = num_strokes - 1;
        hi = strokes[n].raw_center + strokes[n].width / 2;
        /* length of the strokes -- if stroke is too short we */
        /* will not treat the glyph as a flat top/bottom; it */
        /* wil be treated as 'pointy' */
        len_lo = stroke_length(strokes + 0, x);
        len_hi = stroke_length(strokes + n, x);

#ifdef DO_STATS

        /* collect stats for flat/pointy top/bottom's */
        if (len_hi > min_len && ABS(hi - max) < 16)
        {
            FS_PRINTF(("ft 0x%04X %d\n", unicode, hi >> 6));
            flat_things[(hi >> 6) + 256]++;
        }
        else
        {
            FS_PRINTF(("pt 0x%04X %d\n", unicode, max >> 6));
            pointy_things[(max >> 6) + 256]++;
        }
        if (len_lo > min_len && ABS(lo - min) < 16)
        {
            FS_PRINTF(("fb 0x%04X %d\n", unicode, lo >> 6));
            flat_things[(lo >> 6) + 256]++;
        }
        else
        {
            FS_PRINTF(("pb 0x%04X %d\n", unicode, min >> 6));
            pointy_things[(min >> 6) + 256]++;
        }
#endif

        /* ? flat topped character */
        if (len_hi > min_len && ABS(hi - max) < 48)
        {
            /* ? near flat_top */
            if (ABS(hi - rl[1]) < CJK_TROUGH)
            {
#if defined(RTGAH_DEBUG) && (RTGAH_DEBUG & 8192)
                FS_PRINTF(("hi %f near ftop %f snap_to %f\n", hi / 64.0 , rl[1] / 64.0, rrl[1] / 64.0));
#endif
                max = hi;
                new_max = rrl[1];
            }
        }
        else /* pointy top character */
        {
            /* ? near pointy_top */
            if (ABS(max - rl[0]) < CJK_TROUGH)
            {
#if defined(RTGAH_DEBUG) && (RTGAH_DEBUG & 8192)
                FS_PRINTF(("max %f near ptop %f snap_to %f\n", max / 64.0 , rl[0] / 64.0, rrl[0] / 64.0));
#endif
                new_max = rrl[0];
            }
        }
#if defined(RTGAH_DEBUG) && (RTGAH_DEBUG & 8192)
        if (max == new_max)
            FS_PRINTF(("nothing snapped on top\n"));
#endif


        /* ? flat bottomed character */
        if (len_lo > min_len && ABS(lo - min) < 16)
        {
            /* ? near flat_bottom */
            if (ABS(lo - rl[2]) < CJK_TROUGH)
            {
#if defined(RTGAH_DEBUG) && (RTGAH_DEBUG & 8192)
                FS_PRINTF(("lo %f near fbot %f snap_to %f\n", lo / 64.0 , rl[2] / 64.0, rrl[2] / 64.0));
#endif
                min = lo;
                new_min = rrl[2];
            }
        }
        else /* pointy bottomed character */
        {
            /* ? near pointy_bottom */
            if (ABS(min - rl[3]) < CJK_TROUGH)
            {
#if defined(RTGAH_DEBUG) && (RTGAH_DEBUG & 8192)
                FS_PRINTF(("min %f near pbot %f snap_to %f\n", min / 64.0 , rl[3] / 64.0, rrl[3] / 64.0));
#endif
                new_min = rrl[3];
            }
        }
#if defined(RTGAH_DEBUG) && (RTGAH_DEBUG & 8192)
        if (min == new_min)
            FS_PRINTF(("nothing snapped on bottom\n"));
#endif
    }

    /* ? scale character to new size -- to hit new TOP & BOT */
    /* since the character is often built of overlapping pieces */
    /* we can't simply move one of the pieces -- it might no longer overlap */
    /* so we need to scale the character so that the top and bottom */
    /* coincide with where we want them to be -- and everything else */
    /* is evenly stretched/shrunk between those extremes */
    if (max != new_max || min != new_min)
    {
        FS_FIXED s = FixDiv(new_max - new_min, max - min);
        for (i = 0; i < np; i++)
            y[i] = new_min + FixMul(s, oy[i] - min);

        /* adjust the stroke center */
        for (i = 0; i < num_strokes; i++)
        {
            si = strokes + i;
            p0 = y[si->p0];
            p2 = y[si->p2];
            si->raw_center = si->adj_center = (p0 + p2) / 2;
        }
    }

    /* if it's pointy ... still try and snap top/bottom strokes */
    if (new_max == rrl[0] || new_min == rrl[3])
    {
        F26DOT6 c, w;

        /* ? any bottom flats which should align with BOT */
        for (i = 0; i < num_strokes; i++)
        {
            c = strokes[i].raw_center;
            w = strokes[i].width / 2;
            lo = c - w;
            if (ABS(lo - rl[1]) < CJK_TROUGH)
                strokes[i].adj_center = rrl[1] + w;
            else
                break;
        }

        /* ? any top flats that should align with TOP */
        for (i = num_strokes - 1; i >= 0; i--)
        {
            c = strokes[i].raw_center;
            w = strokes[i].width / 2;
            hi = c + w;
            if (ABS(hi - rl[1]) < CJK_TROUGH)
                strokes[i].adj_center = rrl[1] - w;
            else
                break;
        }
    }

#ifdef RTGAH_DEBUG
    if (RTGAH_DEBUG & 8192)
    {
        dump_element("after snap_top_bottom", elementPtr);
        dump_strokes("", num_strokes, strokes);
    }
#endif
    SYS_MEMCPY(oy, y, 4 * np);
}

/********************************************************************************/
static int strokes_overlap(int a, int b, STROKE *strokes)
{
    STROKE *s;
    F26DOT6 La, Ra, Wa, Ta, Ba, Lb, Rb, Wb, Tb, Bb;
    F26DOT6 over_o, over_c, w, c;

    /* if disjoint in "other", they do not overlap */
    if (strokes[a].min > strokes[b].max)
        return 0;
    if (strokes[b].min > strokes[a].max)
        return 0;

    /* extremes of stroke[a] */
    s = strokes + a;
    Wa = s->width;
    c = s->adj_center;
    Ta = c - Wa / 2;
    Ba = c + Wa / 2;
    La = s->min;
    Ra = s->max;

    /* extremes of stroke[b] */
    s = strokes + b;
    Wb = s->width;
    c = s->adj_center;
    Tb = c - Wb / 2;
    Bb = c + Wb / 2;
    Lb = s->min;
    Rb = s->max;

    /* overlap wrt 'other' */
    over_o = MIN(Ra - Lb, Rb - La);

    /* overlap wrt 'coord' */
    over_c = MIN(Ta - Bb, Tb - Ba);

    /* 1/2 the minimum stroke width */
    w = MIN(Wa, Wb) / 2;

    if (over_o < w && over_c < w)
        return 0;   /* overlap is acceptable */

    return 1;   /* not... */
}

/********************************************************************************/
/* ? can thinning strokes make them appear separate  */
static int can_thin_strokes(int a, int b, STROKE *strokes)
{
    STROKE *sa, *sb;
    F26DOT6 wa, wb, ca, cb;

    sa = strokes + a;
    sb = strokes + b;
    /* save for restoration purposes */
    wa = sa->width;
    wb = sb->width;
    ca = sa->adj_center;
    cb = sb->adj_center;

    /* note: (4+7*54)/8 = 47.75 ... so strokes won't get thinner than 3/4 pixel */
    if (wa <= 54 && wb <= 54)
        return 0;

    if (wa > 54)
    {
        sa->width = (4 + 7 * wa) >> 3;
        sa->adj_center = simple_alignment(ca, sa->width);
    }
    if (wb > 54)
    {
        sb->width = (4 + 7 * wb) >> 3;
        sb->adj_center = simple_alignment(cb, sb->width);
    }

    if (strokes_too_close(a, b, strokes))
    {
        /* restore saved values */
        sa->adj_center = ca;
        sb->adj_center = cb;
        sa->width = wa;
        sb->width = wb;
        return 0;
    }
    else
        return 1;
}

/********************************************************************************/
/* ? do strokes appear too close */
static int strokes_too_close(int a, int b, STROKE *strokes)
{
    int r;
    F26DOT6 a_top, b_bot, ia, ib, u, v;

    /* a stroke can't be too close to itself */
    if (a == b)
        return 0;

    if (strokes[a].adj_center > strokes[b].adj_center)
    {
        r = a;
        a = b;
        b = r;
    }

    /* now stroke[a].adj_center < stroke[b].adj_center */
    /* so .. we expect stroke[a] below stroke[b] */

    /* top of the lower edge and bottom  of the upper edge */
    a_top = strokes[a].adj_center + strokes[a].width / 2;
    b_bot = strokes[b].adj_center - strokes[b].width / 2;

    /* overlapping is definately too close */
    if (a_top > b_bot)
        return 1;

    /* the pixels they reside in */
    ia = FLOOR_26_6(a_top);
    ib = FLOOR_26_6(b_bot);

    /* one of 4 cases ... <r> means we're OK */
    if (a_top == ia)
    {
        /* both pixel aligned .. need at least a pixel between */
        if (b_bot == ib)
        {
            r = (b_bot - a_top) >= ONE_26_6;
        }
        else /* one pixel aligned, one not ... need 3/4 pixel between */
        {
            r = (b_bot - a_top) >= 48;
        }
    }
    else
    {
        /* one pixel aligned, one not ... need 3/4 pixel between */
        if (b_bot == ib)
        {
            r = (b_bot - a_top) >= 48;
        }
        else /* neither aligned... */
        {
            /* if the edges are in the same pixel */
            if (ia == ib)
            {
                /* need 3/4 pixels between */
                r = (b_bot - a_top) >= 48;
            }
            /* if edges in adjacent different pixels */
            else if (ib == (ia + 64))
            {
                int c;
                /* need one of the pixels to be < 25% gray, to be OK */
                u = (b_bot - ib) >= 48;   /* upper pixel under 25% gray    */
                v = (a_top - ia) <= 16;   /* lower pixel under 25% gray */
                r = u || v;

                c = (b_bot - ib) >= 40 && (a_top - ia) <= 24;
                /* OR both under 37.5% */
                r |= c;
            }
            else
                r = 1;  /* they are far enough apart */
        }
    }
    return !r;
}

/********************************************************************************/
#ifdef RTGAH_DEBUG
static void indent_fn(int n)
{
    while (n--)
        FS_PRINTF(("- "));
}
#endif /* RTGAH_DEBUG */
/********************************************************************************/
/* <a,b> collided, can we move <b> up ? */
static int can_move_stroke_up_2(
#ifdef RTGAH_DEBUG
                                int indent,
#endif
                                int a, int b, int num_strokes, STROKE *strokes, F26DOT6 *coord, F26DOT6 *other)
{
    /* if b is a top stroke
     *    return 0
     * move to align other edge of b
     * not enough ?
     *    move by whole pixel
     * can we move b alone
     *    do that
     * else can we move b's subtree ?
     *    do that
     * else
     *    fail
     */

    F26DOT6 move, ac[MAX_STROKES];
    int r, z, all = 1;
    FS_BYTE any = 0, too_close[MAX_STROKES] = {0};
#ifdef RTGAH_DEBUG
    if (RTGAH_DEBUG & 8)
    {
        indent_fn(indent);
        FS_PRINTF(("trying to move stroke %d up\n", b));
    }
#endif  /* RTGAH_DEBUG */

    if (num_strokes <= 0)
        return 0;

    /* if b is a top stroke, return 0 */
    z = num_strokes - 1;
    if (strokes[b].raw_center == strokes[z].raw_center)
    {
#ifdef RTGAH_DEBUG
        if (RTGAH_DEBUG & 8)
        {
            indent_fn(indent);
            FS_PRINTF(("sorry %d is a top stroke\n", b));
        }
#endif  /* RTGAH_DEBUG */

        return 0;
    }

    /* save adj_center data */
    for (z = 0; z < num_strokes; z++)
        ac[z] = strokes[z].adj_center;

    /* move to align other edge of b */
    {
        F26DOT6 m, w, top, bot, f;

        m = strokes[b].adj_center;
        w = (1 + strokes[b].width) / 2;
        top = m + w;
        bot = m - w;
        f = FLOOR_26_6(top);

        if (ABS(top - f) < 3)
            move = CEILING_26_6(bot) - bot;
        else
            move = CEILING_26_6(top) - top;
    }

#ifdef RTGAH_DEBUG
    if (move && (RTGAH_DEBUG & 8))
    {
        indent_fn(indent);
        FS_PRINTF(("trying up %7.3f\n", move / 64.0));
    }
#endif  /* RTGAH_DEBUG */

    /* not enough - move by a whole pixel */
    strokes[b].adj_center += move;
    if (move == 0 || strokes_too_close(a, b, strokes))
    {
        move = 64;
        strokes[b].adj_center = ac[b] + move;
#ifdef RTGAH_DEBUG
        if (RTGAH_DEBUG & 8)
        {
            indent_fn(indent);
            FS_PRINTF(("trying up %7.3f\n", move / 64.0));
        }
#endif  /* RTGAH_DEBUG */
    }

    /* but ... don't move stroke[b] by more than 1.5 pixels from original position */
    if (ABS(strokes[b].adj_center - strokes[b].raw_center) > 96)
    {
        strokes[b].adj_center = ac[b];
#ifdef RTGAH_DEBUG
        if (RTGAH_DEBUG & 8)
        {
            indent_fn(indent);
            FS_PRINTF(("too far from original position\n"));
        }
#endif  /* RTGAH_DEBUG */
        return 0;
    }

    /* and not above the top stroke */
    if (strokes[b].adj_center > strokes[num_strokes - 1].adj_center)
    {
        strokes[b].adj_center = ac[b];
#ifdef RTGAH_DEBUG
        if (RTGAH_DEBUG & 8)
        {
            indent_fn(indent);
            FS_PRINTF(("not above the top stroke\n"));
        }
#endif  /* RTGAH_DEBUG */
        return 0;
    }

    /* test all the stuff above b, against b */
    for (z = 0; z < num_strokes; z++)
    {
        too_close[z] = strokes_too_close(b, z, strokes) && strokes_overlap(b, z, strokes);

#ifdef RTGAH_DEBUG
        if (RTGAH_DEBUG & 8)
        {
            indent_fn(indent);
            FS_PRINTF(("wrt %d is %d too close == %d\n", b, z, too_close[z]));
        }
#endif  /* RTGAH_DEBUG */
        any |= too_close[z];
    }

    /* if nothing was too close to the new b ... we're cool */
    if (!any)
    {
#ifdef RTGAH_DEBUG
        if (RTGAH_DEBUG & 8)
        {
            indent_fn(indent);
            FS_PRINTF(("nothing was too close to %d\n", b));
        }
#endif  /* RTGAH_DEBUG */
        r = 1;
    }
    else
    {
        all = 1;    /* so far */
        for (z = b + 1; all && z < num_strokes; z++)
        {
            if (!too_close[z])
                continue;

#ifdef RTGAH_DEBUG
            if (RTGAH_DEBUG & 8)
            {
                indent_fn(indent);
                FS_PRINTF(("trying the hard way on %d\n", z));
            }
#endif  /* RTGAH_DEBUG */
            all &= can_move_stroke_up(b, z, num_strokes, strokes, coord, other);
        }

        if (all)
        {
#ifdef RTGAH_DEBUG
            if (RTGAH_DEBUG & 8)
            {
                indent_fn(indent);
                FS_PRINTF(("moved %d up the hard way\n", b));
            }
#endif /* RTGAH_DEBUG */
            r = 1;
        }
        else
        {
            /* nope, restore everyones center and give up */
            for (z = b; z < num_strokes; z++)
                strokes[z].adj_center = ac[z];
#ifdef RTGAH_DEBUG
            if (RTGAH_DEBUG & 8)
            {
                indent_fn(indent);
                FS_PRINTF(("move %d up failed\n", b));
            }
#endif  /* RTGAH_DEBUG */
            r = 0;
        }
    }

    return r;
}

/********************************************************************************/
/* <a,b> collided, can we move <a> down ? */
static int can_move_stroke_down_2(
#ifdef RTGAH_DEBUG
                                  int indent,
#endif
                                  int a, int b, int num_strokes, STROKE *strokes, F26DOT6 *coord, F26DOT6 *other)
{
    /* if a is a bottom stroke
     *    return 0
     * move to align other edge of a
     * not enough ?
     *    move by whole pixel
     * can we move a alone ?
     *    do that
     * else can we move a's subtree ?
     *    do that
     * else
     *    fail
     */

    F26DOT6 move, ac[MAX_STROKES];

    int r, z, all = 1;
    FS_BYTE any = 0, too_close[MAX_STROKES] = {0};

#ifdef RTGAH_DEBUG
    if (RTGAH_DEBUG & 8)
    {
        indent_fn(indent);
        FS_PRINTF(("trying to move stroke %d down\n", a));
    }
#endif  /* RTGAH_DEBUG */

    if (num_strokes <= 0)
        return 0;

    /* if a is a bottom stroke, return 0 */
    if (strokes[a].adj_center == strokes[0].adj_center)
    {
#ifdef RTGAH_DEBUG
        if (RTGAH_DEBUG & 8)
        {
            indent_fn(indent);
            FS_PRINTF(("sorry %d is a bottom stroke\n", a));
        }
#endif  /* RTGAH_DEBUG */
        return 0;
    }

    /* save adj_center data */
    for (z = 0; z < num_strokes; z++)
        ac[z] = strokes[z].adj_center;

    /* move to align other edge of a */
    {
        F26DOT6 m, w, top, bot, f;

        m = strokes[a].adj_center;
        w = (1 + strokes[a].width) / 2;
        top = m + w;
        bot = m - w;
        f = FLOOR_26_6(top);

        if (ABS(top - f) < 3)
            move = bot - FLOOR_26_6(bot);
        else
            move = top - FLOOR_26_6(top);

#ifdef RTGAH_DEBUG
        if (move && (RTGAH_DEBUG & 8))
        {
            indent_fn(indent);
            FS_PRINTF(("trying down %7.3f\n", move / 64.0));
        }
#endif  /* RTGAH_DEBUG */
    }

    /* not enough - move by a whole pixel */
    strokes[a].adj_center -= move;
    if (move == 0 || strokes_too_close(a, b, strokes))
    {
        move = 64;
        strokes[a].adj_center = ac[a] - move;
#ifdef RTGAH_DEBUG
        if (RTGAH_DEBUG & 8)
        {
            indent_fn(indent);
            FS_PRINTF(("trying down %7.3f\n", move / 64.0));
        }
#endif  /* RTGAH_DEBUG */
    }

    /* but ... don't move stroke[a] by more than 1.5 pixels from original position */
    if (ABS(strokes[a].adj_center - strokes[a].raw_center) > 96)
    {
#ifdef RTGAH_DEBUG
        if (RTGAH_DEBUG & 8)
        {
            indent_fn(indent);
            FS_PRINTF(("too far from original position\n"));
        }
#endif  /* RTGAH_DEBUG */
        strokes[a].adj_center = ac[a];
        return 0;
    }

    /* new for pass 4 */
    if (strokes[a].adj_center < strokes[0].raw_center)
    {
#ifdef RTGAH_DEBUG
        if (RTGAH_DEBUG & 8)
        {
            indent_fn(indent);
            FS_PRINTF(("can't move below bottom stroke\n"));
        }
#endif  /* RTGAH_DEBUG */
        strokes[a].adj_center = ac[a];
        return 0;
    }

    /* test all the stuff below a, against a  */
    for (z = num_strokes - 1; z >= 0; z--)
    {
        too_close[z] = strokes_too_close(a, z, strokes) && strokes_overlap(a, z, strokes);
        any |= too_close[z];

#ifdef RTGAH_DEBUG
        if (RTGAH_DEBUG & 8)
        {
            indent_fn(indent);
            FS_PRINTF(("wrt %d, is %d too close ? %d\n", a, z, too_close[z]));
        }
#endif  /* RTGAH_DEBUG */
    }

    /* if nothing was too close to the new a ... we're cool */
    if (!any)
    {
#ifdef RTGAH_DEBUG
        if (RTGAH_DEBUG & 8)
        {
            indent_fn(indent);
            FS_PRINTF(("nothing was too close to %d\n", a));
        }
#endif  /* RTGAH_DEBUG */
        r = 1;
    }
    else
    {
        /* try moving (down) the stuff that was too close to a ... */
        all = 1;
        for (z = a - 1; all && z >= 0; z--)
        {
            if (!too_close[z])
                continue;

#ifdef RTGAH_DEBUG
            if (RTGAH_DEBUG & 8)
            {
                indent_fn(indent);
                FS_PRINTF(("trying the hard way on %d\n", z));
            }
#endif  /* RTGAH_DEBUG */

            all &= can_move_stroke_down(z, a, num_strokes, strokes, coord, other);
        }

        /* ? did they all move successfully */
        if (all)
        {
            r = 1;
#ifdef RTGAH_DEBUG
            if (RTGAH_DEBUG & 8)
            {
                indent_fn(indent);
                FS_PRINTF(("moved %d down the hard way\n", a));
            }
#endif  /* RTGAH_DEBUG */
        }
        else
        {
            /* nope, restore everyones center and give up */
            for (z = 0; z < num_strokes; z++)
                strokes[z].adj_center = ac[z];
            r = 0;
#ifdef RTGAH_DEBUG
            if (RTGAH_DEBUG & 8)
            {
                indent_fn(indent);
                FS_PRINTF(("move %d down failed\n", a));
            }
#endif  /* RTGAH_DEBUG */
        }
    }

    return r;
}

/********************************************************************************/
/* <a,b> collided, can we move <b> up ? */
static int can_move_stroke_up(int a, int b, int num_strokes, STROKE *strokes, F26DOT6 *coord, F26DOT6 *other)
{
    STROKE *si, *sb;
    FS_USHORT all[MAX_STROKES] = {0};
    F26DOT6 ac[MAX_STROKES];
    FS_USHORT i, num;
    int ok;

    /* save adjusted centers */
    for (i = 0; i < num_strokes; i++)
        ac[i] = strokes[i].adj_center;

    /* other strokes which should be moved along with <b> */
    sb = strokes + b;
    for (i = 0, num = 0; i < num_strokes; i++)
    {
        /* share an edge, or have (almost) the same raw_center */
        si = strokes + i;
        if (sb->p0 == si->p0 || sb->p2 == si->p2 ||
                sb->p0 == si->p2 || sb->p2 == si->p0 ||
                ABS(sb->raw_center - si->raw_center) < 16)
            all[num++] = i;
    }

#ifdef RTGAH_DEBUG
    if (RTGAH_DEBUG & 8)
    {
        FS_PRINTF(("move %d up, requires moving ", b));
        for (i = 0; i < num; i++)
            FS_PRINTF(("%d ", all[i]));
        FS_PRINTF(("up\n"));
    }
#endif /* RTGAH_DEBUG */

    /* can we move <all> strokes up */
    for (ok = 1, i = 0; ok && i < num; i++)
    {
        ok = can_move_stroke_up_2(
#ifdef RTGAH_DEBUG
            0,
#endif
            a, all[i], num_strokes, strokes, coord, other);
    }

    if (!ok)
    {
        /* no ... restore the centers */
        for (i = 0; i < num_strokes; i++)
            strokes[i].adj_center = ac[i];
    }
    return ok;
}

/********************************************************************************/
/* <a,b> collided, can we move <a> down ? */
static int can_move_stroke_down(int a, int b, int num_strokes, STROKE *strokes, F26DOT6 *coord, F26DOT6 *other)
{
    STROKE *si, *sa;
    FS_USHORT all[MAX_STROKES] = {0};
    F26DOT6 ac[MAX_STROKES];
    FS_USHORT i, num;
    int ok;

    /* save adjusted centers */
    for (i = 0; i < num_strokes; i++)
        ac[i] = strokes[i].adj_center;

    /* other strokes which should be moved along with <a> */
    sa = strokes + a;
    for (i = 0, num = 0; i < num_strokes; i++)
    {
        /* share an edge, or have (almost) the same raw_center */
        si = strokes + i;
        if (sa->p0 == si->p0 || sa->p2 == si->p2 ||
                sa->p0 == si->p2 || sa->p2 == si->p0 ||
                ABS(sa->raw_center - si->raw_center) < 16)
            all[num++] = i;
    }

#ifdef RTGAH_DEBUG
    if (RTGAH_DEBUG & 8)
    {
        FS_PRINTF(("move %d down, requires moving ", b));
        for (i = 0; i < num; i++)
            FS_PRINTF(("%d ", all[i]));
        FS_PRINTF(("down\n"));
    }
#endif /* RTGAH_DEBUG */

    /* can we move <all> strokes */
    for (ok = 1, i = 0; ok && i < num; i++)
    {
        ok = can_move_stroke_down_2(
#ifdef RTGAH_DEBUG
                                    0,
#endif
                                    all[i], b, num_strokes, strokes, coord, other);
    }

    if (!ok)
    {
        /* no ... restore all the centers */
        for (i = 0; i < num_strokes; i++)
            strokes[i].adj_center = ac[i];
    }

    return ok;
}

/********************************************************************************/
static void no_collisions(char xy, int num_strokes, STROKE *strokes, fnt_ElementType *elementPtr)
{
    int i, j;
    F26DOT6 *coord, *other, a, b;

    /* first do a simple alignment */
    for (i = 0; i < num_strokes; i++)
        strokes[i].adj_center = simple_alignment(strokes[i].raw_center, strokes[i].width);

#ifdef RTGAH_DEBUG
    if (xy == 'y' && (RTGAH_DEBUG & 8))
        dump_y_strokes("before collisions", num_strokes, strokes, elementPtr);
    if (xy == 'x' && (RTGAH_DEBUG & 8))
        dump_x_strokes("before collisions", num_strokes, strokes, elementPtr);
#endif /* RTGAH_DEBUG */

    i = (xy == 'x');
    coord = i ? elementPtr->x : elementPtr->y;
    other = i ? elementPtr->y : elementPtr->x;

    for (i = 0; i < num_strokes; i++)
    {
        for (j = 0; j < num_strokes; j++)
        {
            /*** some fast eliminations inline here ***/

            /* ? out of order, continue */
            if (strokes[j].adj_center < strokes[i].adj_center)
                continue;

            /* ? disjoint wrt 'other', continue */
            if (strokes[j].min > strokes[i].max)
                continue;
            if (strokes[i].min > strokes[j].max)
                continue;

            /* ? bottom of stroke_j more than 2 pixels above top of stroke_i, continue */
            a = strokes[j].adj_center - strokes[j].width / 2; /* bottom of j */
            b = strokes[i].adj_center + strokes[i].width / 2; /* top of i */
            if ((a - b) > 128)
                continue;

            /*** now, the more complicated (and accurate) tests ***/
            if (strokes_overlap(i, j, strokes))
            {
                if (strokes_too_close(i, j, strokes))
                {
#ifdef RTGAH_DEBUG
                    if (RTGAH_DEBUG & 8)
                        FS_PRINTF(("strokes too close %d %d\n", i, j));
#endif /* RTGAH_DEBUG */

                    /* ? move stroke <i> up  */
                    if (can_move_stroke_up(i, j, num_strokes, strokes, coord, other))
                    {
                        /* no body it's all done in the condition */
#ifdef RTGAH_DEBUG
                        if (RTGAH_DEBUG & 8)
                            FS_PRINTF(("moved stroke %d UP\n", j));
#endif /* RTGAH_DEBUG */
                    }
                    /* ? move stroke <j> down */
                    else if (can_move_stroke_down(i, j, num_strokes, strokes, coord, other))
                    {
                        /* no body it's all done in the condition */
#ifdef RTGAH_DEBUG
                        if (RTGAH_DEBUG & 8)
                            FS_PRINTF(("moved stroke %d DOWN\n", i));
#endif /* RTGAH_DEBUG */
                    }
                    else if (can_thin_strokes(i, j, strokes))
                    {
                        /* no body it's all done in the condition */
#ifdef RTGAH_DEBUG
                        if (RTGAH_DEBUG & 8)
                            FS_PRINTF(("thinned strokes %d %d\n", i, j));
#endif /* RTGAH_DEBUG */
                    }
                    else
                    {
#ifdef RTGAH_DEBUG
                        if (RTGAH_DEBUG & 8)
                            FS_PRINTF(("** nothing worked\n"));
#endif /* RTGAH_DEBUG */
                    }
                }
                else
                {
#ifdef RTGAH_DEBUG
                    if (RTGAH_DEBUG & 8)
                        FS_PRINTF(("strokes %d %d are NOT too close\n", i, j));
#endif /* RTGAH_DEBUG */
                }
            }
            else
            {
#ifdef RTGAH_DEBUG
                if (RTGAH_DEBUG & 8)
                    FS_PRINTF(("strokes %d %d do NOT overlap\n", i, j));
#endif /* RTGAH_DEBUG */
            }
        }

#ifdef RTGAH_DEBUG
        if (xy == 'y' && (RTGAH_DEBUG & 8))
        {
            char str[32];
            FS_SPRINTF((str, "after stroke %d", i));
            dump_y_strokes(str, num_strokes, strokes, elementPtr);
        }
        if (xy == 'x' && (RTGAH_DEBUG & 8))
        {
            char str[32];
            FS_SPRINTF((str, "after stroke %d", i));
            dump_x_strokes(str, num_strokes, strokes, elementPtr);
        }
#endif /* RTGAH_DEBUG */
    }

#ifdef RTGAH_DEBUG
    if (RTGAH_DEBUG & 8)
        dump_strokes("after no_collisions", num_strokes, strokes);
#endif /* RTGAH_DEBUG */
}

/********************************************************************************/
/* Given 3 points on a triangle p0,p1,p2; the cross product */
/* (p1-p0) x (p2-p0) is negative if the triangle is clockwise */
static int is_contour_clockwise(int c, fnt_ElementType *elementPtr)
{
    F26DOT6 *x = elementPtr->x;
    F26DOT6 *y = elementPtr->y;
    FS_USHORT *sp = elementPtr->sp;
    FS_USHORT *ep = elementPtr->ep;
    int p1, p0, p2;
    F26DOT6 z;
    register int i, n;

    /* topmost point on contour (leftmost if tie) */
    p1 = sp[c];
    z = y[p1];
    for (i = 1 + p1, n = ep[c]; i <= n; i++)
    {
        if ((y[i] > z) || (y[i] == z && x[i] < x[p1]))
        {
            p1 = i;
            z = y[p1];
        }
    }

    p0 = PREV(p1);
    if (x[p0] == x[p1] && y[p0] == y[p1])
        p0 = PREV(p0);
    p2 = NEXT(p1);
    z = Mul26Dot6(x[p1] - x[p0], y[p2] - y[p0]) - Mul26Dot6(y[p1] - y[p0], x[p2] - x[p0]);

    /* mohammed's bug */
    if (z == 0)
    {
        /* ok the area is too small ... do it the slow way.
         * by analysing the *unit* vectors
         * out from p1 towards p0 (L)
         * and out from p1 towards p2 (R)
         *
         *               R
         *           p1 --- p2
         *          /
         *       L /
         *        /
         *       p0
         *
         * L.x < R.x it's clockwise
        */

        FIXED_VECTOR L, R;

        L.x = (x[p0] - x[p1]) << 10;
        L.y = (y[p0] - y[p1]) << 10;
        fixed_norm(&L);
        R.x = (x[p2] - x[p1]) << 10;
        R.y = (y[p2] - y[p1]) << 10;
        fixed_norm(&R);
        z = L.x - R.x;
    }

    return z < 0;
}


/********************************************************************************/
/* find sequences of points within a contour where coordinate <xy> is constant */
static int find_pieces(char xy, PIECE *pieces, fnt_ElementType *elementPtr,
                       PLIST **pplist, PLIST *plistMax)
{
    F26DOT6 *coords;
    F26DOT6 *other;
    FS_USHORT *ep = elementPtr->ep;
    FS_USHORT *sp = elementPtr->sp;
    FS_USHORT nc = elementPtr->nc;
    FS_USHORT c, start, end, p0, p1, pt;
    int num_pieces = 0;
    int d;

    if (xy == 'x')
    {
        coords = elementPtr->x;
        other = elementPtr->y;
    }
    else
    {
        coords = elementPtr->y;
        other = elementPtr->x;
    }

    for (c = 0; num_pieces < MAX_PIECES && c < nc; c++)
    {
        start = sp[c];
        end = ep[c];

        /* surprise ... there are 1 pt contours out there */
        if (start == end)
            continue;

        p0 = p1 = start;

        /* we allow some slop on the pieces */

        if (ABS(coords[PREV(p0)] - coords[p0]) <= 3)
            p0 = PREV(p0);

        while (p1 < end)
        {
            while (p1 < end && ABS(coords[p1 + 1] - coords[p0]) <= 3)
                p1++;

            if (p0 != p1)
            {
                PIECE *p = pieces + num_pieces++;
                F26DOT6 c0 = (coords[p0] + coords[p1]) / 2;

                p->p0 = p0;
                p->p1 = p1;
                p->c = c;
                p->pts = 0;

                /* fix the slop */
                coords[p0] = c0;
                coords[p1] = c0;

                /* add the other points (if any) to PLIST */
                for (pt = NEXT(p0); pt != p1 ; pt = NEXT(pt))
                {
                    p->pts = add_to_plist(p->pts, new_plist(pt, pplist));
                    coords[pt] = c0; /* fix the slop */
                    if (*pplist >= plistMax)
                        break;
                }

                /* direction */
                d = (other[p1] > other[p0]) ? 1 : -1;
                p->d = (FS_USHORT)d;
                if (num_pieces == MAX_PIECES)
                    break;
            }

            p1++;
            p0 = p1;
        }
    }

    if (num_pieces == MAX_PIECES)
        num_pieces--;

    /* up-sort the pieces by coords[pieces.p0] */
    if (num_pieces)
    {
        int i, j;
        PIECE v;
        F26DOT6 cvp0;

        for (i = 1; i < num_pieces; i++)
        {
            v = pieces[i];
            cvp0 = coords[v.p0];
            for (j = i - 1; j >= 0 && coords[pieces[j].p0] > cvp0; j--)
                pieces[j + 1] = pieces[j];
            pieces[j + 1] = v;
        }
    }

    /* make 'almost' aligned pieces, actually aligned */
    /* thanks to some poorly drawn characters in TBG-bold */
    if (num_pieces)
    {
        int i, j;
        F26DOT6 ci;
        PIECE *pj;
        PLIST *q;

        for (i = 0; i < num_pieces; i++)
        {
            ci = coords[pieces[i].p0];
            for (j = i + 1; j < num_pieces; j++)
            {
                if ((coords[pieces[j].p0] - ci) > 8)
                    break;

                /* pieces are within 1/8 pixel ... make them the same */
                pj = pieces + j;
                coords[pj->p0] = coords[pj->p1] = ci;
                for (q = pj->pts; q; q = q->next)
                    coords[q->pt] = ci;
            }
        }
    }

#ifdef RTGAH_DEBUG
    if (RTGAH_DEBUG & 32)
        dump_pieces("raw_pieces", num_pieces, pieces, elementPtr);
#endif /* RTGAH_DEBUG */

    return num_pieces;
}

/********************************************************************************/
/* find sequences of points within a contour where coordinate <xy> is constant  */
/* this is the CJK version. see non-CJK version above                           */
static int find_pieces_CJK(char xy, PIECE *pieces, fnt_ElementType *elementPtr,
                           PLIST **pplist, PLIST *plistMax)
{
    F26DOT6 *coords;
    F26DOT6 *other;
    FS_USHORT *ep = elementPtr->ep;
    FS_USHORT *sp = elementPtr->sp;
    FS_USHORT nc = elementPtr->nc;
    FS_USHORT c, start, end, p0, p1;
    int num_pieces = 0;
    int d;

    if (xy == 'x')
    {
        coords = elementPtr->x;
        other = elementPtr->y;
    }
    else
    {
        coords = elementPtr->y;
        other = elementPtr->x;
    }

    for (c = 0; num_pieces < MAX_PIECES && c < nc; c++)
    {
        start = sp[c];
        end = ep[c];

        /* surprise ... there are 1 pt contours out there */
        if (start == end)
            continue;

        p0 = p1 = start;
        if (ABS(coords[p1] - coords[PREV(p0)]) <= MAX_PIECE_SLOP)
            p0 = PREV(p0);

        while (p1 < end)
        {
            /* the remaining pieces will not wrap */
            while (p1 < end && ABS(coords[p0] - coords[p1 + 1]) <= MAX_PIECE_SLOP)
                p1++;

            if (p0 != p1)
            {
                F26DOT6 l, w;
                FS_USHORT pt;

                /* the pieces need to be *more or less* H or V */
                /* know w <= MAX_PIECE_SLOP */
                w = coords[p1] - coords[p0];
                if (w < 0)
                    w = -w;
                l = other[p1] - other[p0];
                if (l < 0)
                    l = -l;
                if (ANG_TOL * w < l)
                {
                    PIECE *p = pieces + num_pieces++;

                    p->p0 = p0;
                    p->p1 = p1;
                    p->c = c;
                    p->pts = 0;

                    /* add the other points (if any) to PLIST */
                    for (pt = NEXT(p0); pt != p1 ; pt = NEXT(pt))
                    {
                        p->pts = add_to_plist(p->pts, new_plist(pt, pplist));
                        if (*pplist >= plistMax)
                            break;
                    }
                    d = (other[p1] > other[p0]) ? 1 : -1;
                    p->d = (FS_SHORT)d;
                    if (num_pieces == MAX_PIECES)
                        break;
                }
            }
            p1 = p0 = (1 + p1);
        }
    }

    if (num_pieces == MAX_PIECES)
        num_pieces--;

    if (num_pieces == 0)
        return 0;

    /* up-sort the pieces by coords[pieces.p0] */
    {
        int i, j;
        PIECE v;
        F26DOT6 cvp0;

        for (i = 1; i < num_pieces; i++)
        {
            v = pieces[i];
            cvp0 = coords[v.p0];
            for (j = i - 1; j >= 0 && coords[pieces[j].p0] > cvp0; j--)
                pieces[j + 1] = pieces[j];
            pieces[j + 1] = v;
        }
    }

#ifdef RTGAH_DEBUG
    if (RTGAH_DEBUG & 32)
        dump_pieces("raw_pieces", num_pieces, pieces, elementPtr);
#endif /* RTGAH_DEBUG */

    return num_pieces;
}

/* make strokes :: a stroke is a pair of pieces which delimit "ink" */
static int find_strokes(char xy, int num_pieces, PIECE *pieces, STROKE *strokes, fnt_ElementType *elementPtr,
                        F26DOT6 max_sw, PLIST **pplist, PLIST *plistMax)
{
    int i, j, on = (xy == 'x') ? 1 : -1;
    F26DOT6 *x = (xy == 'x') ? elementPtr->x : elementPtr->y;
    F26DOT6 *y = (xy == 'x') ? elementPtr->y : elementPtr->x;
    PIECE *pi, *pj;
    STROKE *ps;
    F26DOT6 i_lo, i_hi, j_lo, j_hi;
    F26DOT6 l, w;
    int num_strokes = 0;

#if defined(RTGAH_DEBUG) && (RTGAH_DEBUG & 2)
    FS_PRINTF(("\nfind_strokes\n"));
#endif


    /*  find the contour containing the left-most point in the glyph
    *   to determine if the glyph is "ink on left", or "ink on right"
    *   and to correct the direction of ON transitions.
    *   on was set initially for "ink on right" glyphs
    */
    {
        FS_USHORT pt, c, nc = elementPtr->nc;
        FS_USHORT *sp = elementPtr->sp;
        FS_USHORT *ep = elementPtr->ep;
        F26DOT6 *xx = elementPtr->x;
        F26DOT6 min = xx[0];
        FS_USHORT min_c = 0;

        for (c = 0; c < nc; c++)
        {
            /* skip single point contours */
            if (sp[c] == ep[c])
                continue;

            for (pt = sp[c]; pt <= ep[c]; pt++)
            {
                if (xx[pt] < min)
                {
                    min = xx[pt];
                    min_c = c;
                }
            }
        }

        c = (FS_USHORT)is_contour_clockwise(min_c, elementPtr);
        if (!c)
        {
            /* ink on left.  the sign of "on" is backwards */
            on = -on;
        }

#if defined(RTGAH_DEBUG) && (RTGAH_DEBUG & 2)
        FS_PRINTF(("leftmost contour %d clockwise %d\n", min_c, c));
#endif

    }

    /* find the next ON transition and the (hopefully) matching OFF transition */
    for (i = 0; i < num_pieces && num_strokes < MAX_STROKES; i++)
    {
        pi = pieces + i;
        if (pi->d != on)
            continue;

        /* y extents of pi */
        i_lo = y[pi->p0];
        i_hi = y[pi->p1];
        if (i_lo > i_hi)
        {
            F26DOT6 z = i_lo;
            i_lo = i_hi;
            i_hi = z;
        }

        for (j = i + 1; j < num_pieces && num_strokes < MAX_STROKES; j++)
        {
            pj = pieces + j;

            /* potential stroke must be a reasonable width */
            w = x[pj->p0] - x[pi->p0];
            if (w > max_sw)
                break;

            /* pieces must be different directions */
            if (pj->d == on)
            {
                continue;
            }

            /* y extents of pj */
            j_lo = y[pj->p0];
            j_hi = y[pj->p1];
            if (j_lo > j_hi)
            {
                F26DOT6 z = j_lo;
                j_lo = j_hi;
                j_hi = z;
            }

            /***/
            /*** to make *SURE* we have an ON-OFF transition pair ***/
            /*** we need to compute winding numbers .             ***/
            /*** but this would be much too S L O W               ***/
            /***/
            /*** so we rely on some heuristics                    ***/
            /*** that work *ALMOST* all of the time               ***/
            /***/

            /*** a heuristic that USUALLY works and occasionally fails ***/
            /* pieces must overlap in "other" coordinate */
            if (j_lo > i_hi || j_hi < i_lo)
            {
                continue;
            }

            /*** a heuristic that USUALLY works and occasionally fails ***/
            /* if length of overlap  < 1/8 the stroke width ... it's no good */
            l = MIN(i_hi, j_hi) - MAX(i_lo, j_lo);
            if (l < (w >> 3))
            {
                continue;
            }


            /* add the stroke */

#if defined(RTGAH_DEBUG) && (RTGAH_DEBUG & 2)
            FS_PRINTF(("add_stroke %d %d = %d %d %d %d\n", i, j, pi->p0, pi->p1, pj->p0, pj->p1));
#endif
            ps = strokes + num_strokes++;
            ps->p0 = pi->p0;
            ps->p1 = pi->p1;
            ps->p2 = pj->p0;
            ps->p3 = pj->p1;
            ps->min = MIN(i_lo, j_lo);
            ps->max = MAX(i_hi, j_hi);
            ps->e1 = copy_plist(pi->pts, pplist);
            if (*pplist >= plistMax)
                return 0;
            ps->e2 = copy_plist(pj->pts, pplist);
            if (*pplist >= plistMax)
                return 0;
            ps->raw_center = (x[pi->p0] + x[pj->p0]) / 2;
            ps->adj_center = ps->raw_center;
            ps->width = (x[pj->p0] - x[pi->p0]);
        }
    }

    /* sort strokes by increasing <raw_center>
    *  insertion sort is quickest for small N
    */
    {
        STROKE v;
        F26DOT6 sirc;

        for (i = 1; i < num_strokes; i++)
        {
            v = strokes[i];
            sirc = v.raw_center;
            for (j = i - 1; j >= 0 && strokes[j].raw_center > sirc; j--)
                strokes[j + 1] = strokes[j];
            strokes[j + 1] = v;
        }
    }

#if defined(RTGAH_DEBUG) && (RTGAH_DEBUG & 2)
    dump_strokes("initial", num_strokes, strokes);
#endif

    /* combine strokes with same center and width */
    {
        register STROKE *si, *sj;

        for (i = 0; i < num_strokes; i++)
        {
            si = strokes + i;
            for (j = i + 1; j < num_strokes; j++)
            {
                sj = strokes + j;
                if (sj->raw_center > si->raw_center)
                    break;

                if (si->width == sj->width)
                {
                    FS_USHORT z, pts[4];
                    STROKE s = {0};
                    PLIST *p;

                    /* new left edge */
                    pts[0] = si->p0;
                    pts[1] = si->p1;
                    pts[2] = sj->p0;
                    pts[3] = sj->p1;

                    /* make pts[] ascending wrt y[] */
                    if (y[pts[0]] > y[pts[1]])
                    {
                        z = pts[0];
                        pts[0] = pts[1];
                        pts[1] = z;
                    }
                    if (y[pts[2]] > y[pts[3]])
                    {
                        z = pts[2];
                        pts[2] = pts[3];
                        pts[3] = z;
                    }
                    if (y[pts[0]] > y[pts[2]])
                    {
                        z = pts[0];
                        pts[0] = pts[2];
                        pts[2] = z;
                    }
                    if (y[pts[1]] > y[pts[3]])
                    {
                        z = pts[1];
                        pts[1] = pts[3];
                        pts[3] = z;
                    }
                    if (y[pts[1]] > y[pts[2]])
                    {
                        z = pts[1];
                        pts[1] = pts[2];
                        pts[2] = z;
                    }

                    s.p0 = pts[0];
                    s.p1 = pts[3];

                    /* we can USE (rather than COPY) the lists si->e1 and sj->e1 */
                    /* s.e1 = si->e1 append sj->e1                               */
                    s.e1 = p = si->e1;          /* use si->e1 */
                    if (p)
                    {
                        while (p->next)         /* go to last elt */
                            p = p->next;        /* of si->e1      */
                        p->next = sj->e1;       /* append sj->e1  */
                    }
                    else
                        s.e1 = sj->e1;
                    sj->e1 = 0;

                    /* add points 1 and 2 -- if they are new */
                    if (pts[1] != pts[0] && pts[1] != pts[3])
                        s.e1 = add_to_plist(s.e1 , new_plist(pts[1], pplist));
                    if (*pplist >= plistMax)
                        return 0;
                    if (pts[2] != pts[0] && pts[2] != pts[3])
                        s.e1 = add_to_plist(s.e1 , new_plist(pts[2], pplist));
                    if (*pplist >= plistMax)
                        return 0;

                    /* new right edge */
                    pts[0] = si->p2;
                    pts[1] = si->p3;
                    pts[2] = sj->p2;
                    pts[3] = sj->p3;

                    /* make pts[] ascending wrt y[] -- sorting networks, Knuth V3 */
                    if (y[pts[0]] > y[pts[1]])
                    {
                        z = pts[0];
                        pts[0] = pts[1];
                        pts[1] = z;
                    }
                    if (y[pts[2]] > y[pts[3]])
                    {
                        z = pts[2];
                        pts[2] = pts[3];
                        pts[3] = z;
                    }
                    if (y[pts[0]] > y[pts[2]])
                    {
                        z = pts[0];
                        pts[0] = pts[2];
                        pts[2] = z;
                    }
                    if (y[pts[1]] > y[pts[3]])
                    {
                        z = pts[1];
                        pts[1] = pts[3];
                        pts[3] = z;
                    }
                    if (y[pts[1]] > y[pts[2]])
                    {
                        z = pts[1];
                        pts[1] = pts[2];
                        pts[2] = z;
                    }

                    s.p2 = pts[0];
                    s.p3 = pts[3];

                    /* s.e2 = si->e2 append sj->e2 */
                    s.e2 = p = si->e2;
                    if (p)
                    {
                        while (p->next)
                            p = p->next;
                        p->next = sj->e2;
                    }
                    else
                        s.e2 = sj->e2;
                    sj->e2 = 0;

                    /* add points 1 and 2 -- if they are new */
                    if (pts[1] != pts[0] && pts[1] != pts[3])
                        s.e2 = add_to_plist(s.e2 , new_plist(pts[1], pplist));
                    if (*pplist >= plistMax)
                        return 0;
                    if (pts[2] != pts[0] && pts[2] != pts[3])
                        s.e2 = add_to_plist(s.e2 , new_plist(pts[2], pplist));
                    if (*pplist >= plistMax)
                        return 0;

                    s.min = MIN(y[s.p2], y[s.p0]);
                    s.max = MAX(y[s.p3], y[s.p1]);
                    s.raw_center = s.adj_center = si->raw_center;
                    s.width = si->width;
                    strokes[i] = s;
                    sj->width = -MYINFINITY;

#if defined(RTGAH_DEBUG) && (RTGAH_DEBUG & 2)
                    {
                        char str[80];
                        FS_SPRINTF((str, "after combining strokes %d and %d", i, j));
                        dump_strokes(str, num_strokes, strokes);
                    }
#endif
                }
            }
        }
    }


#if defined(RTGAH_DEBUG) && (RTGAH_DEBUG & 2)
    dump_strokes("after combining", num_strokes, strokes);
#endif

    /* if two strokes of different width, share an edge; */
    /* mark for deletion the one with the shorter overlap */
    {
        F26DOT6 si_lo, si_hi, sj_lo, sj_hi, tmp;
        STROKE *si, *sj;

        for (i = 0; i < num_strokes; i++)
        {
            si = strokes + i;
            /* ? si dead, continue */
            if (si->width == -MYINFINITY)
                continue;

            for (j = i + 1; j < num_strokes; j++)
            {
                sj = strokes + j;

                /* ? sj dead, continue */
                if (sj->width == -MYINFINITY)
                    continue;

                /* ? same width, continue */
                if (si->width == sj->width)
                    continue;

                if (x[si->p0] == x[sj->p0])
                {
                    si_lo = si->p0;
                    si_hi = si->p1;
                    sj_lo = sj->p0;
                    sj_hi = sj->p1;
                }

                else if (x[si->p0] == x[sj->p2])
                {
                    si_lo = si->p0;
                    si_hi = si->p1;
                    sj_lo = sj->p2;
                    sj_hi = sj->p3;
                }
                else if (x[si->p2] == x[sj->p0])
                {
                    si_lo = si->p2;
                    si_hi = si->p3;
                    sj_lo = sj->p0;
                    sj_hi = sj->p1;
                }
                else if (x[si->p2] == x[sj->p2])
                {
                    si_lo = si->p2;
                    si_hi = si->p3;
                    sj_lo = sj->p2;
                    sj_hi = sj->p3;
                }
                else
                    continue;    /* no edge overlap */


                /* correct point order */
                if (si_lo > si_hi)
                {
                    tmp = si_lo;
                    si_lo = si_hi;
                    si_hi = tmp;
                }

                if (sj_lo > sj_hi)
                {
                    tmp = sj_lo;
                    sj_lo = sj_hi;
                    sj_hi = tmp;
                }

                /* ? edges disjoint, continue */
                if (si_hi < sj_lo || sj_hi < si_lo)
                    continue;

                /* ok "shared" edges are not disjoint */
                /* mark the shorter stroke as dead    */
                if (stroke_length(si, y) < stroke_length(sj, y))
                    si->width = -MYINFINITY;
                else
                    sj->width = -MYINFINITY;
            }
        }
    }

#if defined(RTGAH_DEBUG) && (RTGAH_DEBUG & 2)
    dump_strokes("after marking", num_strokes, strokes);
#endif


    /* now remove the dead strokes by copying good ones over them
    *  if stroke[k] is dead, strokes[k..last-1] := strokes[k+1 .. last]
    */

    for (i = 0; i < num_strokes; /**/)
    {
        while (i < num_strokes && strokes[i].width != -MYINFINITY)
            i++;

        j = i + 1;
        while (j < num_strokes && strokes[j].width == -MYINFINITY)
            j++;

        if (j < num_strokes)
        {
            strokes[i++] = strokes[j];
            strokes[j].width = -MYINFINITY;
        }
        else
            break;
    }
    num_strokes = i;

#if defined(RTGAH_DEBUG) && (RTGAH_DEBUG & 2)
    dump_strokes("after deletion", num_strokes, strokes);
    if (xy == 'x') dump_x_strokes("after deletion", num_strokes, strokes, elementPtr);
    if (xy == 'y') dump_y_strokes("after deletion", num_strokes, strokes, elementPtr);
    dump_element("", elementPtr);
    FS_PRINTF(("\n"));
#endif

    return num_strokes;
}
/********************************************************************************/
/* make strokes :: a stroke is a pair of pieces which delimit "ink"             */
/* this is the CJK version. see non-CJK version above                           */
static int find_strokes_CJK(char xy, int num_pieces, PIECE *pieces, STROKE *strokes, fnt_ElementType *elementPtr,
                            F26DOT6 max_sw, PLIST **pplist, PLIST *plistMax)
{
    int i, j, on = (xy == 'x') ? 1 : -1;
    F26DOT6 *x = (xy == 'x') ? elementPtr->x : elementPtr->y;
    F26DOT6 *y = (xy == 'x') ? elementPtr->y : elementPtr->x;
    PIECE *pi, *pj;
    STROKE *ps;
    F26DOT6 i_lo, i_hi, j_lo, j_hi;
    F26DOT6 w;
    int num_strokes = 0;

#if defined(RTGAH_DEBUG) && (RTGAH_DEBUG & 2)
    FS_PRINTF(("\nfind_strokes\n"));
#endif

    /*  find the contour containing the left-most point in the glyph
    to determine if the glyph is "ink on left", or "ink on right"
    and to correct the direction of ON transitions.
    on was set initially for "ink on right" glyphs */
    {
        FS_USHORT pt, c, nc = elementPtr->nc;
        FS_USHORT *sp = elementPtr->sp;
        FS_USHORT *ep = elementPtr->ep;
        F26DOT6 *xx = elementPtr->x;
        F26DOT6 min = xx[0];
        FS_USHORT min_c = 0;

        for (c = 0; c < nc; c++)
        {
            for (pt = sp[c]; pt <= ep[c]; pt++)
            {
                if (xx[pt] < min)
                {
                    min = xx[pt];
                    min_c = c;
                }
            }
        }

        c = (FS_USHORT)is_contour_clockwise(min_c, elementPtr);
        if (!c)
        {
            /* ink on left.  the sign of "on" is backwards */
            on = -on;
        }

    }

    /* find the next ON transition and the matching OFF transition */
    for (i = 0; i < num_pieces && num_strokes < MAX_STROKES; i++)
    {
        pi = pieces + i;
        if (pieces[i].d != on)
            continue;

        i_lo = MIN(y[pi->p0], y[pi->p1]);
        i_hi = MAX(y[pi->p0], y[pi->p1]);

        for (j = i + 1; j < num_pieces && num_strokes < MAX_STROKES; j++)
        {
            pj = pieces + j;
            /* potential stroke must be a reasonable width */
            w = x[pj->p0] - x[pi->p0];
            if (w > max_sw)
                break;

            /* pieces must be different directions */
            if (pj->d == on)
                continue;

            /* pieces must overlap in "other" coordinate */
            j_lo = MIN(y[pj->p0], y[pj->p1]);
            j_hi = MAX(y[pj->p0], y[pj->p1]);
            if (j_lo > i_hi || j_hi < i_lo)
                continue;

            /* add the stroke */

#if defined(RTGAH_DEBUG) && (RTGAH_DEBUG & 2)
            FS_PRINTF(("add_stroke %d %d = %d %d %d %d\n", i, j, pi->p0, pi->p1, pj->p0, pj->p1));
#endif
            ps = strokes + num_strokes++;
            ps->p0 = pi->p0;
            ps->p1 = pi->p1;
            ps->p2 = pj->p0;
            ps->p3 = pj->p1;
            ps->min = MIN(i_lo, j_lo);
            ps->max = MAX(i_hi, j_hi);
            ps->e1 = copy_plist(pi->pts, pplist);
            if (*pplist >= plistMax)
                return 0;
            ps->e2 = copy_plist(pj->pts, pplist);
            if (*pplist >= plistMax)
                return 0;
            ps->raw_center = (x[pi->p0] + x[pj->p0]) / 2;
            ps->adj_center = ps->raw_center;
            ps->width = (x[pj->p0] - x[pi->p0]);
        }
    }

    /* sort strokes by increasing <raw_center>
    insertion sort is quickest for small N */
    {
        STROKE v;
        F26DOT6 sirc;

        for (i = 1; i < num_strokes; i++)
        {
            v = strokes[i];
            sirc = v.raw_center;
            for (j = i - 1; j >= 0 && strokes[j].raw_center > sirc; j--)
                strokes[j + 1] = strokes[j];
            strokes[j + 1] = v;

        }
    }


    /* if two strokes of different width, share an edge; */
    /* mark for deletion the one with the shorter overlap */
    {
        F26DOT6 a, b, c, d, lo, hi, Li, Lj;
        STROKE *si, *sj;

        for (i = 0; i < num_strokes; i++)
        {
            si = strokes + i;
            /* ? si dead, continue */
            if (si->width == -MYINFINITY)
                continue;

            for (j = i + 1; j < num_strokes; j++)
            {
                sj = strokes + j;

                /* ? sj dead, continue */
                if (sj->width == -MYINFINITY)
                    continue;

                if (si->width == sj->width)
                    continue;

                /* ? don't share an edge */
                if ((si->p0 != sj->p0 || si->p1 != sj->p1) &&
                        (si->p2 != sj->p2 || si->p3 != sj->p3))
                    continue;

                /* length of overlap in stroke[i] other coord; */
                a = y[si->p0];
                b = y[si->p1]; /* lower edge */
                c = y[si->p2];
                d = y[si->p3]; /* higher edge */
                hi = MIN(MAX(a, b), MAX(c, d));
                lo = MAX(MIN(a, b), MIN(c, d));
                Li = hi - lo;

                /* length of overlap in stroke[j] other coord; */
                a = y[sj->p0];
                b = y[sj->p1]; /* lower edge */
                c = y[sj->p2];
                d = y[sj->p3]; /* higher edge */
                hi = MIN(MAX(a, b), MAX(c, d));
                lo = MAX(MIN(a, b), MIN(c, d));
                Lj = hi - lo;

                /* mark the shorter overlap for deletion */
                if (Li < Lj)
                    si->width = -MYINFINITY;
                else
                    sj->width = -MYINFINITY;
            }
        }
    }


#if defined(RTGAH_DEBUG) && (RTGAH_DEBUG & 2)
    dump_strokes("after marking", num_strokes, strokes);
    dump_element("after marking", elementPtr);
#endif

    /* combine strokes with same center and width */
    {
        for (i = 0; i < num_strokes; i++)
        {
            if (strokes[i].width == -MYINFINITY)
                continue;
            for (j = i + 1; j < num_strokes; j++)
            {
                if (strokes[j].raw_center > strokes[i].raw_center)
                    break;

                if (strokes[j].width == -MYINFINITY)
                    continue;

                if (strokes[i].width == strokes[j].width)
                {
                    FS_USHORT pts[4];
                    STROKE s = {0};
                    PLIST *p;

                    /* new left edge */
                    pts[0] = strokes[i].p0;
                    pts[1] = strokes[i].p1;
                    pts[2] = strokes[j].p0;
                    pts[3] = strokes[j].p1;

                    /* make pts[] ascending wrt y[] */
                    {
                        int m, n;
                        FS_USHORT pm;
                        F26DOT6 ym;

                        for (m = 1; m < 4; m++)
                        {
                            pm = pts[m];
                            ym = y[pm];
                            for (n = m - 1; n >= 0 && y[pts[n]] > ym; n--)
                                pts[n + 1] = pts[n];
                            pts[n + 1] = pm;
                        }
                    }
                    s.p0 = pts[0];
                    s.p1 = pts[3];
                    s.e1 = 0;

                    /* add points 1 and 2 -- if they are new */
                    if (pts[1] != pts[0] && pts[1] != pts[3])
                        s.e1 = add_to_plist(s.e1 , new_plist(pts[1], pplist));
                    if (*pplist >= plistMax)
                        return 0;
                    if (pts[2] != pts[0] && pts[2] != pts[3])
                        s.e1 = add_to_plist(s.e1 , new_plist(pts[2], pplist));
                    if (*pplist >= plistMax)
                        return 0;

                    /* add the named pts */
                    for (p = strokes[i].e1; p; p = p->next)
                        s.e1 = add_to_plist(s.e1 , new_plist(p->pt, pplist));
                    if (*pplist >= plistMax)
                        return 0;
                    for (p = strokes[j].e1; p; p = p->next)
                        s.e1 = add_to_plist(s.e1 , new_plist(p->pt, pplist));
                    if (*pplist >= plistMax)
                        return 0;

                    /* new right edge */
                    pts[0] = strokes[i].p2;
                    pts[1] = strokes[i].p3;
                    pts[2] = strokes[j].p2;
                    pts[3] = strokes[j].p3;

                    /* make pts[] ascending wrt y[] */
                    {
                        int m, n;
                        FS_USHORT pm;
                        F26DOT6 ym;

                        for (m = 1; m < 4; m++)
                        {
                            pm = pts[m];
                            ym = y[pm];
                            for (n = m - 1; n >= 0 && y[pts[n]] > ym; n--)
                                pts[n + 1] = pts[n];
                            pts[n + 1] = pm;
                        }
                    }
                    s.p2 = pts[0];
                    s.p3 = pts[3];

                    /* add points 1 and 2 -- if they are new */
                    s.e2 = 0;

                    if (pts[1] != pts[0] && pts[1] != pts[3])
                        s.e2 = add_to_plist(s.e2 , new_plist(pts[1], pplist));
                    if (*pplist >= plistMax)
                        return 0;
                    if (pts[2] != pts[0] && pts[2] != pts[3])
                        s.e2 = add_to_plist(s.e2 , new_plist(pts[2], pplist));
                    if (*pplist >= plistMax)
                        return 0;

                    /* add the named points */
                    for (p = strokes[i].e2; p; p = p->next)
                        s.e2 = add_to_plist(s.e2 , new_plist(p->pt, pplist));
                    if (*pplist >= plistMax)
                        return 0;
                    for (p = strokes[j].e2; p; p = p->next)
                        s.e2 = add_to_plist(s.e2 , new_plist(p->pt, pplist));
                    if (*pplist >= plistMax)
                        return 0;

                    s.min = MIN(y[s.p2], y[s.p0]);
                    s.max = MAX(y[s.p3], y[s.p1]);
                    s.raw_center = s.adj_center = strokes[i].raw_center;
                    s.width = strokes[i].width;
                    strokes[i] = s;
                    strokes[j].width = -MYINFINITY;

#if defined(RTGAH_DEBUG) && (RTGAH_DEBUG & 2)
                    {
                        char str[80];
                        FS_SPRINTF((str, "after combining strokes %d and %d", i, j));
                        dump_strokes(str, num_strokes, strokes);
                    }
#endif
                }
            }
        }
    }

#if defined(RTGAH_DEBUG) && (RTGAH_DEBUG & 2)
    dump_strokes("after combining", num_strokes, strokes);
#endif

    /* now remove the dead strokes by copying good ones over them
    if stroke[k] is dead, strokes[k..last-1] := strokes[k+1 .. last] */

    for (i = 0; i < num_strokes; /**/)
    {
        while (i < num_strokes && strokes[i].width != -MYINFINITY)
            i++;

        j = i + 1;
        while (j < num_strokes && strokes[j].width == -MYINFINITY)
            j++;

        if (j < num_strokes)
        {
            strokes[i++] = strokes[j];
            strokes[j].width = -MYINFINITY;
        }
        else
            break;
    }
    num_strokes = i;

#if defined(RTGAH_DEBUG) && (RTGAH_DEBUG & 2)
    dump_strokes("after deletion", num_strokes, strokes);
    FS_PRINTF(("\n"));
#endif
    return num_strokes;
}

/********************************************************************************/
static F26DOT6 simple_alignment(F26DOT6 center, F26DOT6 width)
{
    F26DOT6 v0, v1, d0, d1, d;

    v0 = center - (width >> 1);
    v1 = center + (width >> 1);
    d0 = ROUND_26_6(v0) - v0;
    d1 = ROUND_26_6(v1) - v1;
    d = (ABS(d0) < ABS(d1)) ? d0 : d1;
    return center + d;
}

/********************************************************************************/
/* These macros are unused
#define same_side(p,z,n) ((p<z && n<z) || (p>z && n>z))
#define CACHE_SIZE 8 / * must be a power of 2 * /
#define CACHE_AND  (CACHE_SIZE - 1)
*/

/********************************************************************************/
static int get_edges(int num_strokes, STROKE *strokes, F26DOT6 *x, EDGE *edges)
{
    int num_edges = 0;
    int i;
    EDGE *e;
    STROKE *s;

    for (i = 0; i < num_strokes; i++)
    {
        /* first edge of stroke[i] */
        s = strokes + i;
        e = edges + num_edges++;
        e->pt = s->p0;
        e->x = x[s->p0];
        e->y0 = s->min;
        e->y1 = s->max;

        /* fake strokes only have 1 edge */
        if (s->p0 == s->p2)
            continue;

        /* second edge of stroke[i] */
        e = edges + num_edges++;
        e->pt = s->p2;
        e->x = x[s->p2];
        e->y0 = s->min;
        e->y1 = s->max;
    }

    /* sort the edges by increasing x .. should be nearly sorted already */
    {
        int j;
        EDGE v;
        F26DOT6 vx;

        for (i = 1; i < num_edges; i++)
        {
            v = edges[i];
            vx = v.x;
            for (j = i - 1; j >= 0 && edges[j].x > vx; j--)
                edges[j + 1] = edges[j];
            edges[j + 1] = v;
        }
    }

    return num_edges;
}

/********************************************************************************/
static void find_above_below(F26DOT6 px, F26DOT6 py, int num_edges, EDGE *edges, int *p_above, int *p_below)
{
    int start, index, above, below;
    F26DOT6 d, d_above, d_below;
    register EDGE *e;


    /* find closest above */
    d_above = MYINFINITY;
    above = -1;
    e = edges;

    /*** find the smallest index :: e[index].x >= px ... call it start */
    /* ? can we start looking in the middle */
    index = num_edges / 2;
    if (edges[index].x >= px)
        index = 0; /* nope */

    while (e[index].x < px)
        index++;
    start = index;

    /* process the edges above <px> until they are too far above <px> .. or we run out of edges */
    do
    {
        e = edges + index;

        /* perpendicular distance from edge to point */
        d = e->x - px;

        /* ? potential improvement */
        if (d < d_above)
        {
            /* maybe add a parallel component to the distance */
            /* ? off left side of edge, add distance to p0 */
            if (py < e->y0)
                d += (e->y0 - py);
            /* ? off right side of edge, add distance to p1  */
            else if (py > e->y1)
                d += (py - e->y1);

            /* ? definite improvement */
            if (d < d_above)
            {
                d_above = d;
                above = index;
            }
        }
        else
            break;
        index++;
    }
    while (index < num_edges);
    *p_above = edges[above].pt;

    /* find closest below */
    d_below = MYINFINITY;
    below = -1;
    e = edges;

    /* we want the largest index :: e[index].x <= px */
    /* it can't be far from start, which is the smallest value :: e[start].x >= px */
    index = start;
    while (e[index].x > px)
        index--;

    /*** process the edges below <px> until they are too far below ... or we run out of edges. */
    do
    {
        e = edges + index;
        d = px - e->x;

        /* ? potential improvement */
        if (d < d_below)
        {
            if (py < e->y0)
                d += (e->y0 - py);
            else if (py > e->y1)
                d += (py - e->y1);

            /* ? definite improvement */
            if (d < d_below)
            {
                d_below = d;
                below = index;
            }
        }
        else
            break;
        index--;
    }
    while (index >= 0);
    *p_below = edges[below].pt;
}
/********************************************************************************/
/* apply changes made to strokes to the outline data */
static int adjust_outline_1(char xy, int num_strokes, STROKE *strokes, fnt_ElementType *elementPtr)
{
    F26DOT6 *hinted;
    FS_BYTE *f = elementPtr->f, touched;
    int i, j;

    if (xy == 'x')
    {
        hinted = elementPtr->x;
        touched = XMOVED;
    }
    else
    {
        hinted = elementPtr->y;
        touched = YMOVED;
    }

    /* initial grid alignment of strokes  */
    for (i = 0; i < num_strokes; i++)
    {
        STROKE *si = strokes + i;

        if (si->adj_center == si->raw_center)
            si->adj_center = simple_alignment(si->adj_center, si->width);
    }

#if defined(RTGAH_DEBUG) && (RTGAH_DEBUG & 4)
    dump_strokes("after simple_alignment", num_strokes, strokes);
#endif


    /* move and touch points on the strokes */
    for (i = 0; i < num_strokes; i++)
    {
        PLIST *p;
        F26DOT6 z;
        int p0, p1, p2, p3;
        STROKE *si;

        si = strokes + i;

        /* if any of the points on the stroke are already touched -- bail */
        if ((f[si->p0] | f[si->p1] | f[si->p2] | f[si->p3]) & touched)
            continue;

        /* move the endpoints of edge 1 */
        z = si->adj_center - si->width / 2;
        p0 = si->p0;
        p1 = si->p1;
        hinted[p0] = z;
        f[p0] |= touched;
        hinted[p1] = z;
        f[p1] |= touched;

        /* move and touch the remaining points on edge 1 */
        for (p = si->e1; p; p = p->next)
        {
            j = p->pt;
            f[j] |= touched;
            hinted[j] = z;
        }

        /* move the endpoints of edge 2 */
        z = si->adj_center + si->width / 2;
        p2 = si->p2;
        p3 = si->p3;
        hinted[p2] = z;
        f[p2] |= touched;
        hinted[p3] = z;
        f[p3] |= touched;

        /* move and touch the remaining points on edge 2 */
        for (p = si->e2; p; p = p->next)
        {
            j = p->pt;
            f[j] |= touched;
            hinted[j] = z;
        }
    }

#if defined(RTGAH_DEBUG) && (RTGAH_DEBUG & 4)
    dump_element("after adjust_outline_1", elementPtr);
#endif

    return num_strokes;
}

/********************************************************************************/
/* apply changes made to strokes to the outline data */
static int adjust_outline_2(char xy, int num_strokes, STROKE *strokes, fnt_ElementType *elementPtr)
{
    F26DOT6 *scaled, *hinted, *other;
    FS_BYTE *f = elementPtr->f, touched;
    FS_USHORT *ep = elementPtr->ep;
    FS_USHORT nc = elementPtr->nc;
    int np = 1 + ep[nc - 1];

    if (xy == 'x')
    {
        hinted = elementPtr->x;
        scaled = elementPtr->ox;
        other = elementPtr->oy;
        touched = XMOVED;
    }
    else
    {
        hinted = elementPtr->y;
        scaled = elementPtr->oy;
        other = elementPtr->ox;
        touched = YMOVED;
    }

    /* add fake strokes at glyph max and glyph min */
    /* makes the subsequent interpolation phase simpler */
    {
        STROKE *next;
        int xmin_pt, xmax_pt;
        F26DOT6 xmin, xmax, ymin, ymax;
        register int i;
        register F26DOT6 z;

        xmin_pt = xmax_pt = 0;
        xmin = xmax = scaled[0];
        ymin = ymax = other[0];
        for (i = 1; i < np; i++)
        {
            z = other[i];
            if (z < ymin)
            {
                ymin = z;
            }
            else if (z > ymax)
            {
                ymax = z;
            }

            /* hinted points replace unhinted ones */
            z = scaled[i];
            if (z == xmin && (f[i] & touched))
            {
                xmin_pt = i;
                continue;
            }
            if (z == xmax && (f[i] & touched))
            {
                xmax_pt = i;
                continue;
            }
            if (z < xmin)
            {
                xmin = z;
                xmin_pt = i;
            }
            else if (z > xmax)
            {
                xmax = z;
                xmax_pt = i;
            }
        }

        /* add a full height vertical stroke on left (assuming xy=='x') */
        next = strokes + num_strokes++;
        next->p0 = next->p1 = next->p2 = next->p3 = (FS_USHORT)xmin_pt;
        next->raw_center = xmin;
        next->adj_center = hinted[xmin_pt]; /* different from xmin if the point is hinted */
        next->min = ymin;
        next->max = ymax;
        next->width = 1;
        next->e1 = 0;
        next->e2 = 0;

        /* add a full height vertical stroke on right (assuming xy=='x') */
        next = strokes + num_strokes++;
        next->p0 = next->p1 = next->p2 = next->p3 = (FS_USHORT)xmax_pt;
        next->raw_center = xmax;
        next->adj_center = hinted[xmax_pt];
        next->min = ymin;
        next->max = ymax;
        next->width = 1;
        next->e1 = 0;
        next->e2 = 0;

        /* re-sort strokes by ascending raw center. they are almost in order so this is fast */
        {
            int j;
            STROKE v;
            F26DOT6 sirc;   /* Stroke I Raw Center */

            for (i = 1; i < num_strokes; i++)
            {
                v = strokes[i];
                sirc = v.raw_center;
                for (j = i - 1; j >= 0 && strokes[j].raw_center > sirc; j--)
                    strokes[j + 1] = strokes[j];
                strokes[j + 1] = v;
            }
        }
    }

#if defined(RTGAH_DEBUG) && (RTGAH_DEBUG & 4)
    dump_element("after adjust_outline_2", elementPtr);
#endif

    return num_strokes;
}

/********************************************************************************/
/* CJK version of interpolate contour extrema */
/* faster but less flexible than the 'normal' version */
static int adjust_outline_3_CJK(char xy, int num_strokes, STROKE *strokes, fnt_ElementType *elementPtr)
{
    F26DOT6 *scaled, *hinted, *other;
    FS_BYTE *f = elementPtr->f, touched;
    FS_USHORT *sp = elementPtr->sp;
    FS_USHORT *ep = elementPtr->ep;
    FS_USHORT c, nc = elementPtr->nc;

    if (xy == 'x')
    {
        hinted = elementPtr->x;
        scaled = elementPtr->ox;
        other = elementPtr->oy;
        touched = XMOVED;
    }
    else
    {
        hinted = elementPtr->y;
        scaled = elementPtr->oy;
        other = elementPtr->ox;
        touched = YMOVED;
    }

    /* make sure the extrema of each contour are touched */
    for (c = 0; c < nc; c++)
    {
        FS_USHORT min, max, pt, extrema[2], ei;
        F26DOT6 si, oi;
        FS_USHORT p0, p2;
        F26DOT6 u, v, w;
        int i, j, k;

        if (sp[c] == ep[c])
            continue;

        /* find the extrema for this contour */
        min = max = pt = sp[c];
        for (/**/; pt <= ep[c]; pt++)
        {
            if (scaled[pt] < scaled[min]) min = pt;
            else if (scaled[pt] > scaled[max]) max = pt;
        }
        extrema[0] = max;
        extrema[1] = min;

        /* find stroke edges on either side of the extrema */
        /* and interpolate the position of the extrema */
        for (i = 0; i < 2; i++)
        {
            STROKE *sj;

            ei = extrema[i];
            if (f[ei] & touched) continue;

            si = scaled[ei];
            oi = other[ei];

            for (j = num_strokes - 1, sj = strokes + j; j >= 0; j--, sj--)
            {
                STROKE *sk;

                /* point should be just to the right of left edge, and within the vertical range (assuming xy=='x') */
                if (si < scaled[sj->p0] || oi < sj->min || oi > sj->max)
                    continue;

                for (k = j + 1, sk = strokes + k; k < num_strokes; k++, sk++)
                {
                    /* point should be just to left of right edge, and within the vertical range (assuming xy=='x') */
                    if (si > scaled[sk->p2] || oi < sk->min || oi > sk->max)
                        continue;

                    /* point[i] is between strokes ... interpolate  */
                    /* */
                    /*    hinted[i] - hinted[sj->p0]       scaled[i] - scaled[sj->p0] */
                    /* ------------------------------- = ------------------------------- */
                    /* hinted[sk->p2] - hinted[sj->p0]   scaled[sk->p2] - scaled[sj->p0] */

                    p0 = sj->p0;
                    p2 = sk->p2;

                    u = hinted[p2] - hinted[p0];
                    v = si - scaled[p0];
                    w = scaled[p2] - scaled[p0];

#if defined(RTGAH_DEBUG) && (RTGAH_DEBUG & 128)
                    /***/ FS_PRINTF(("contour extrema %d between %d and %d\n", ei, p0, p2));
#endif
                    /* if u==0 or v==0 the LongMulDiv==0 */
                    /* if w==0 then scaled[i] == scaled[p2] == scaled[p0] */
                    if (u == 0 || v == 0 || w == 0)
                        hinted[ei] = hinted[p0];
                    else
                        hinted[ei] = hinted[p0] + LongMulDiv(u, v, w);
                    f[ei] |= touched;

                    /* exit both the "for (j=" loop AND the "for (k=" loops */
                    j = -1;
                    k = num_strokes;
                }
            }
        }
    }

#if defined(RTGAH_DEBUG) && (RTGAH_DEBUG & 4)
    dump_element("after adjust_outline_2", elementPtr);
#endif

    return num_strokes;
}

/********************************************************************************/
/* interpolate contour extrema (for non-CJK glyphs) between nearest edges */
static int adjust_outline_3(char xy, int num_strokes, STROKE *strokes, fnt_ElementType *elementPtr, fnt_LocalGraphicStateType *gs)
{
    F26DOT6 *scaled, *hinted, *other;
    FS_BYTE *f = elementPtr->f, touched;
    FS_USHORT *sp = elementPtr->sp;
    FS_USHORT *ep = elementPtr->ep;
    FS_USHORT c, nc = elementPtr->nc;
    EDGE *edges = (EDGE *)gs->globalGS->rtgah_data.RTGAHEdges;
    int num_edges;
    int last_above = MYINFINITY;
    int last_below = MYINFINITY;
    FS_FIXED scl = 0;

    if (xy == 'x')
    {
        hinted = elementPtr->x;
        scaled = elementPtr->ox;
        other = elementPtr->oy;
        touched = XMOVED;
    }
    else
    {
        hinted = elementPtr->y;
        scaled = elementPtr->oy;
        other = elementPtr->ox;
        touched = YMOVED;
    }

    /* interpolate contour extrema by nearest edges */
    /* get distinct edges from the strokes, and sort them by "x" */
    num_edges = get_edges(num_strokes, strokes, scaled, edges);

    for (c = 0; c < nc; c++)
    {
        FS_USHORT prev, pt, next;
        int above, below;
        register F26DOT6 u, v, w;

        if (sp[c] == ep[c])
            continue;

        for (prev = ep[c], pt = sp[c]; pt <= ep[c]; prev = pt, pt++)
        {
            if (f[pt] & touched)
                continue;

            next = NEXT(pt);


            /* ? is the pt a local min or local max
            *
            * having both terms having <= (or >=) was letting through
            * all of the points '*' on an H/V line  _ _ _ * * *  _ _ _
            * making one of them < (or >) will only let through one endpoint _ _ _ - - * _ _ _
            *
            * if ((scaled[prev] <= scaled[pt] && scaled[next] < scaled[pt]) ||
            *     (scaled[prev] >= scaled[pt] && scaled[next] > scaled[pt]))
            */

            u = scaled[prev];
            v = scaled[pt];
            w = scaled[next];

            if ((u <= v && w < v) || (u >= v && w > v))
            {
                find_above_below(scaled[pt], other[pt], num_edges, edges, &above, &below);

                if (above < 0 || below < 0)
                {
                    /* shouldn't happen */
                    continue;
                }


                /* the interpolation is of this form
                *
                *   hinted[pt] - hinted[below]      scaled[pt] - scaled[below]
                * ----------------------------- = -----------------------------
                * hinted[above] - hinted[below]   scaled[above] - scaled[below]
                *
                * hinted[pt] = hinted[below] + LongMulDiv(u,v,w) ... where
                */
                u = scaled[pt] - scaled[below];
                v = hinted[above] - hinted[below];
                w = scaled[above] - scaled[below];

                /* no silly calculations */
                if (u == 0 || v == 0 || w == 0)
                    hinted[pt] = hinted[below];
                else
                {
                    /* ? can we re-use scl */
                    if (last_above != above || last_below != below)
                    {
                        /* nope */
                        last_above = above;
                        last_below = below;
                        scl = FixDiv(v, w);
                    }
                    hinted[pt] = hinted[below] + FixMul(u, scl);
                }
                f[pt] |= touched;
            }
        }
    }

#if defined(RTGAH_DEBUG) && (RTGAH_DEBUG & 4)
    dump_element("after adjust_outline_3", elementPtr);
#endif

    return num_strokes;
}

/* adjust CFF contour endpoints in y if one has been hinted */
static FS_VOID adjust_outline_cff(char xy, fnt_ElementType *elementPtr)
{
    F26DOT6 *hinted;
    FS_USHORT *sp = elementPtr->sp;
    FS_USHORT *ep = elementPtr->ep;
    FS_USHORT c, nc = elementPtr->nc;

    if (xy == 'x')
        return; /* only close contours in y to avoid rendering problems */
    else
        hinted = elementPtr->y;

    /* make sure endpoints of each contour still join */
    for (c = 0; c < nc; c++)
    {
        if (hinted[ep[c]] == hinted[ep[c] - 1])
            hinted[sp[c]] = hinted[ep[c]];
        else
            hinted[ep[c]] = hinted[sp[c]];
    }
}

/*****************************************************************************/

/* This is ununsed
#define SAME_SIDE(a,b,c) ((scaled[a]<scaled[b] && scaled[c]<scaled[b]) || (scaled[a]>scaled[b] && scaled[c]>scaled[b]))
*/

/*****************************************************************************/
static int adjust_outline(char xy, int script, int num_strokes, STROKE *strokes, fnt_ElementType *elementPtr, fnt_LocalGraphicStateType *gs)
{
    int r;

    r = adjust_outline_1(xy, num_strokes, strokes, elementPtr);
    r = adjust_outline_2(xy, r, strokes, elementPtr);
    if (script == SCRIPT_CJK)
    {
        r = adjust_outline_3_CJK(xy, r, strokes, elementPtr);
        if (gs->globalGS->rtgah_data.fnt_type == CFF_TYPE)
            adjust_outline_cff(xy, elementPtr);
    }
    else
        r = adjust_outline_3(xy, r, strokes, elementPtr, gs);

    return r;
}
/*****************************************************************************/
/* we should have a < b  ... are they far enough apart */
static int visually_separate(F26DOT6 a, F26DOT6 b)
{
    F26DOT6 fa, fb, cb;

    /* this shouldn't happen */
    if (a >= b)
        return 0;

    fa = FLOOR_26_6(a);
    fb = FLOOR_26_6(b);
    cb = CEILING_26_6(b);

    /* ? in same pixel */
    if (fa == fb)
    {
        /* just BARELY in same pixel is ok */
        return a - fa < 16 && cb - b < 16;
    }

    /* ? adjacent pixels */
    if (fa + 64 == fb)
    {
        /* they can intrude a little more into adjacent pixels */
        return a - fa < 32 && cb - b < 32;
    }

    /* they are at least a whole pixel apart ... that's ok */
    return 1;
}

/*lint -e644 Warning -- Variable may not have been initialized */
/********************************************************************************/
/* make sure multi-dot accents are distinct */
/* note, neither fnt_IUP(), not gace() modifies f[] */
/* so unless the contours have a hint/stroke - they will be untouched. */
static void multi_dot(fnt_ElementType *elementPtr)
{
    F26DOT6 *ox = elementPtr->ox;
    F26DOT6 *oy = elementPtr->oy;
    F26DOT6 *x = elementPtr->x;
    F26DOT6 *y = elementPtr->y;
    FS_BYTE *f = elementPtr->f;
    FS_USHORT *ep = elementPtr->ep;
    FS_USHORT *sp = elementPtr->sp;
    FS_USHORT c, nc = elementPtr->nc;
    FS_USHORT start, end;
    FS_USHORT xmin, ymin, xmax, ymax;
    BOOJUM contours[4];
    int i, num = 0;
    FS_BYTE z;
    F26DOT6 tmp;

#ifdef RTGAH_DEBUG
    if (RTGAH_DEBUG & 64)
    {
        dump_element("before multi-dot()", elementPtr);
    }
#endif /* RTGAH_DEBUG */

    if (nc == 0 )
        return;

    /* un-hinted contour extrema */
    for (c = 0; c < nc; c++)
    {
        start = sp[c];
        end = ep[c];
        if (start == end)
            continue;

        xmin = xmax = ymin = ymax = start;
        z = 0;
        for (i = start + 1; i <= end; i++)
        {
            /* use un-hinted coords for this */
            if (ox[i] > ox[xmax]) xmax = (FS_USHORT)i;
            if (ox[i] < ox[xmin]) xmin = (FS_USHORT)i;
            if (oy[i] > oy[ymax]) ymax = (FS_USHORT)i;
            if (oy[i] < oy[ymin]) ymin = (FS_USHORT)i;
            z |= f[i];
        }

        /* ? contour untouched AND contour small enough to qualify (3 pixels x 3 pixels) */
        if ((!z) && (ox[xmax] - ox[xmin]) < 192 && (oy[ymax] - oy[ymin]) < 192)
        {
#ifdef RTGAH_DEBUG
            if (RTGAH_DEBUG & 64)
            {
                FS_PRINTF(("adding contour %d\n", c));
            }
#endif /* RTGAH_DEBUG */

            contours[num].c = c;
            contours[num].xmax = xmax;
            contours[num].xmin = xmin;
            tmp = (ox[xmax] + ox[xmin]) / 2;
            contours[num].oxc = tmp;
            contours[num].xc = tmp;
            contours[num].ymax = ymax;
            contours[num].ymin = ymin;
            tmp = (oy[ymax] + oy[ymin]) / 2;
            contours[num].oyc = tmp;
            contours[num].yc = tmp;
            num++;
            if (num == 4)   /* never more than 4 dots right ? */
                break;
        }
    }

    if ( (num == 0) || (num >= 4))
        return;

    /* make similarly aligned contours, EXACTLY aligned contours */
    for (i = 0; i < num; i++)
    {
        F26DOT6 xc_i, yc_i;
        int j;

        xc_i = contours[i].xc;
        yc_i = contours[i].yc;
        for (j = i + 1; j < num; j++)
        {
            if (ABS(xc_i - contours[j].xc) < 32)
            {
#ifdef RTGAH_DEBUG
                if (RTGAH_DEBUG & 64)
                {
                    FS_PRINTF(("xc %d := xc %d\n", j, i));
                }
#endif /* RTGAH_DEBUG */
                contours[j].xc = xc_i;
            }
            if (ABS(yc_i - contours[j].yc) < 32)
            {
#ifdef RTGAH_DEBUG
                if (RTGAH_DEBUG & 64)
                {
                    FS_PRINTF(("yc %d := yc %d\n", j, i));
                }
#endif /* RTGAH_DEBUG */
                contours[j].yc = yc_i;
            }
        }
    }

    /* round contour centers to half grid (pixel centers) */
    for (i = 0; i < num; i++)
    {
        F26DOT6 m;
        m = contours[i].xc;
        m = FLOOR_26_6(m) + 32;
        contours[i].xc = m;
        m = contours[i].yc;
        m = FLOOR_26_6(m) + 32;
        contours[i].yc = m;
    }

    /* move each contour onto its new pixel center */
    for (i = 0; i < num; i++)
    {
        int j;
        F26DOT6 dx, dy;

        dx = contours[i].xc - contours[i].oxc;
        dy = contours[i].yc - contours[i].oyc;
        j = contours[i].c;
        start = sp[j];
        end = ep[j];
        for (j = start; j <= end; j++)
        {
            x[j] = ox[j] + dx;
            y[j] = oy[j] + dy;
            f[j] |= (XMOVED | YMOVED);
        }
    }

#ifdef RTGAH_DEBUG
    if (RTGAH_DEBUG & 64)
    {
        dump_element("after round centers", elementPtr);
    }
#endif /* RTGAH_DEBUG */


#ifdef RTGAH_DEBUG
    if (RTGAH_DEBUG & 64)
    {
        dump_contours("before x-sort", num, contours);
    }
#endif /* RTGAH_DEBUG */

    if (num < 2)
        return;

    /* sort by increasing x[xmax] */
    {
        int  j, k, h, len = num;
        BOOJUM v, *a = contours;
        int incs[6] = { 112, 48, 21, 7, 3, 1 };
        for (k = 0; k < 6; k++)
        {
            for (i = h = incs[k]; i < len; i++)
            {
                v = a[i];
                j = i;
                while (j >= h && x[a[j - h].xmax] > x[v.xmax])
                {
                    a[j] = a[j - h];
                    j -= h;
                }
                a[j] = v;
            }
        }
    }

#ifdef RTGAH_DEBUG
    if (RTGAH_DEBUG & 64)
    {
        dump_contours("after x-sort", num, contours);
    }
#endif /* RTGAH_DEBUG */

    /* make sure that the candidates are separate in x[] */
    for (c = 0; c < num; c++)
    {
        int a, b, j, k;

        for (i = c + 1; i < num; i++)
        {
            /* if contours are separate in ox[] */
            a = contours[c].xmax;   /* max of lefthand contour */
            b = contours[i].xmin;   /* min of righthand contour */
            if (ox[a] < ox[b])
            {
                if (!visually_separate(x[a], x[b]))
                {
                    /* move one on the right, more to the right */
                    j = contours[i].c;
                    start = sp[j];
                    end = ep[j];
                    for (k = start; k <= end; k++)
                        x[k] += 64;
                    contours[i].xc += 64;
#ifdef RTGAH_DEBUG
                    if (RTGAH_DEBUG & 64)
                    {
                        FS_PRINTF(("moved contour %d wrt contour %d\n", contours[i].c, contours[c].c));
                        dump_contours("after x-move", num, contours);
                        dump_element("after x-move", elementPtr);
                    }
#endif /* RTGAH_DEBUG */
                }
            }
        }
    }

#ifdef RTGAH_DEBUG
    if (RTGAH_DEBUG & 64)
    {
        dump_element("after separate in x", elementPtr);
    }
#endif /* RTGAH_DEBUG */

    /* sort by increasing y[ymax] */
    {
        int  j, k, h, len = num;
        BOOJUM v, *a = contours;
        int incs[6] = { 112, 48, 21, 7, 3, 1 };
        for (k = 0; k < 6; k++)
        {
            for (i = h = incs[k]; i < len; i++)
            {
                v = a[i];
                j = i;
                while (j >= h && y[a[j - h].ymax] > y[v.ymax])
                {
                    a[j] = a[j - h];
                    j -= h;
                }
                if (j < 4 )
                    a[j] = v;
            }
        }
    }

#ifdef RTGAH_DEBUG
    if (RTGAH_DEBUG & 64)
    {
        dump_contours("after y-sort", num, contours);
    }
#endif /* RTGAH_DEBUG */

    /* make sure that the candidates are separate in y[] */
    for (c = 0; c < num; c++)
    {
        int a, b, j, k;

        for (i = c + 1; i < num; i++)
        {
            /* ? are separate in oy[] */
            a = contours[c].ymax;
            b = contours[i].ymin;
            if (oy[a] < oy[b])
            {
                if (!visually_separate(y[a], y[b]))
                {
                    j = contours[i].c;
                    start = sp[j];
                    end = ep[j];
                    for (k = start; k <= end; k++)
                        y[k] += 64;
                    contours[i].yc += 64;
#ifdef RTGAH_DEBUG
                    if (RTGAH_DEBUG & 64)
                    {
                        FS_PRINTF(("moved contour %d wrt contour %d\n", contours[i].c, contours[c].c));
                        dump_contours("after y-move", num, contours);
                        dump_element("after y-move", elementPtr);
                    }
#endif /* RTGAH_DEBUG */
                }
            }
        }
    }

#ifdef RTGAH_DEBUG
    if (RTGAH_DEBUG & 64)
    {
        dump_element("after separate in y", elementPtr);
    }
#endif /* RTGAH_DEBUG */

    /* special case for the Arabic 3 dot component */
    if (num == 3)
    {
        int e = 0;
        F26DOT6 oa, ob, a, b, d = 0;

        /* was one contour roughly centered wrt ox[] ? */
        oa = contours[0].oxc;
        ob = (contours[1].oxc + contours[2].oxc) / 2;
        if (ABS(oa - ob) < 32)
        {
            /* and NOT roughly centered wrt x[] */
            a = contours[0].xc;
            b = (contours[1].xc + contours[2].xc) / 2;
            if (a - b > 32)
            {
                /* a is too far right */
                e = 0;
                d = -64;
            }
            else if (b - a > 32)
            {
                /* a is too far left */
                e = 0;
                d = 64;
            }
        }

        /* was one contour roughly centered wrt ox[] ? */
        oa = contours[1].oxc;
        ob = (contours[0].oxc + contours[2].oxc) / 2;
        if (ABS(oa - ob) < 32)
        {
            /* and NOT roughly centered wrt x[] */
            a = contours[1].xc;
            b = (contours[0].xc + contours[2].xc) / 2;
            if (a - b > 32)
            {
                /* a is too far right */
                e = 1;
                d = -64;
            }
            else if (b - a > 32)
            {
                /* a is too far left */
                e = 1;
                d = 64;
            }
        }

        /* was one contour roughly centered wrt ox[] ? */
        oa = contours[2].oxc;
        ob = (contours[0].oxc + contours[1].oxc) / 2;
        if (ABS(oa - ob) < 32)
        {
            /* and NOT roughly centered wrt x[] */
            a = contours[2].xc;
            b = (contours[0].xc + contours[1].xc) / 2;
            if (a - b > 32)
            {
                /* a is too far right */
                e = 2;
                d = -64;
            }
            else if (b - a > 32)
            {
                /* a is too far left */
                e = 2;
                d = 64;
            }
        }

        /* something to move ? */
        if (d != 0)
        {
            int j;

            j = contours[e].c;
            start = sp[j];
            end = ep[j];
            for (j = start; j <= end; j++)
                x[j] += d;
        }
    }

#ifdef RTGAH_DEBUG
    if (RTGAH_DEBUG & 64)
    {
        dump_element("after multi-dot", elementPtr);
    }
#endif /* RTGAH_DEBUG */
}

/* !!! as defined in fnt.c !!! */
FS_VOID fnt_IUP(fnt_LocalGraphicStateType *gs);

/********************************************************************************/
/* adjust the position of horizontal and vertical strokes in a glyph */
/* optionally: try and avoid colliding strokes */
/* optionally: make nice Latin side bearings and accents */
void rtgah(fnt_LocalGraphicStateType *gs)
{
    fnt_ElementType *elementPtr = gs->CE0;
    int ppm = gs->globalGS->pixelsPerEm;
    F26DOT6 *ref_lines = gs->globalGS->sfnt->ref_lines;
    PIECE *pieces = (PIECE *)gs->globalGS->rtgah_data.RTGAHPieces;
    STROKE *strokes = (STROKE *)gs->globalGS->rtgah_data.RTGAHStrokes;
    int script = gs->globalGS->rtgah_data.script;
    PLIST *plist = 0, *plistMax = 0;
    int n_pieces;
    int n_strokes;
    F26DOT6 max_sw;
    int do_latin_hints;
    int do_stroke_collisions;
    int do_multi_dot;
    int is_italic;
    F26DOT6 rl4[4];
    F26DOT6 latin_rl[6];

    if (elementPtr->nc == 0)
        return;

    /* ? in a B+W rendering ... quit */
    if (!(gs->globalGS->rtgah_data.stateflags & FLAGS_GRAYSCALE))
        return;

    /* ? in a stroke font ... quit */
    if (gs->globalGS->rtgah_data.fontflags & FONTFLAG_STIK)
        return;

    /* copy the Latin ref lines in case we modify them */
    SYS_MEMCPY(latin_rl, ref_lines, 6 * sizeof(F26DOT6));

    is_italic = (gs->globalGS->rtgah_data.fontflags & FONTFLAG_IS_ITALIC);
    do_stroke_collisions = (script == SCRIPT_CJK) || (script == SCRIPT_HANGUL);
    do_multi_dot = (script == SCRIPT_ARABIC) || (script == SCRIPT_HEBREW);
    do_latin_hints = (script == SCRIPT_LATIN) || (script == SCRIPT_GREEK) || (script == SCRIPT_CYRILLIC);

    /* we can't do latin hints if there are no latin ref_lines */
    if (do_latin_hints && ref_lines[RTGA_CAP_SQUARE] == 0)
        do_latin_hints = 0;

#ifdef Y_BOOST
    /*
    ** re-scale lowercase at small ppm, to boost the x-height by a pixel
    ** also scale uppercase to be at least two pixels above x-height
    ** NOTE: the scaling is not isomorphic, only points above the baseline
    ** are scaled so that descenders do not get too large
    ** NOTE: it also makes the lowercase a little bolder -- usually not a bad thing
    */
    if (do_latin_hints && ppm < 19)
    {
        /* always bump the x-height */
        F26DOT6 xh  = ref_lines[RTGA_LC_ROUND];
        int i, np = 1 + elementPtr->ep[elementPtr->nc - 1];
        FS_FIXED s = FixDiv(xh + 8, xh);
        F26DOT6 *y = elementPtr->y;
        F26DOT6 *oy = elementPtr->oy;
#ifdef X_BOOST
        F26DOT6 *x = elementPtr->x;
        F26DOT6 *ox = elementPtr->ox;
#endif

        /* modify the (latin) ref_lines */
        for (i = 0; i < 6; i++)
        {
            if (ref_lines[i] > 0)
                ref_lines[i] = FixMul(s, ref_lines[i]);
        }

        /* check that lower case ref lines don't round differently */
        /* as a result of scaling                                  */
        if (ROUND_26_6(ref_lines[RTGA_LC_ROUND]) != ROUND_26_6(ref_lines[RTGA_LC_SQUARE]))
            ref_lines[RTGA_LC_SQUARE] = ref_lines[RTGA_LC_ROUND];

        /* check that the cap-height ref line is at least two pixels */
        /* above x-height ref line                                   */
        if (ref_lines[RTGA_CAP_SQUARE] - ref_lines[RTGA_LC_SQUARE] < 128)
            ref_lines[RTGA_CAP_SQUARE] = ref_lines[RTGA_LC_SQUARE] + 128;

        /* modify the outline data */
        for (i = 0; i < np; i++)
        {
#ifdef X_BOOST
            x[i] = ox[i] = FixMul(s, ox[i]);
#endif
            if (oy[i] > 0)
                y[i] = oy[i] = FixMul(s, oy[i]);
        }

#ifdef X_BOOST
        {
            /* modify the right side bearing */
            F26DOT6 rsb = ox[np + RIGHTSIDEBEARING];

            rsb = FixMul(s, rsb);
            ox[np + RIGHTSIDEBEARING] = rsb;
            x[np + RIGHTSIDEBEARING] = ROUND_26_6(rsb);
        }
#endif

    } /* do_latin_hints and ppm < 19 */

#endif  /* X and Y BOOST */

    /* generic fix ups to account for sloppy drawing, */
    /* so that we actually find edges (and hence strokes) */
    /* at non-corner extrema */
    non_corner_extrema(elementPtr);

    /* set the maximum stroke width */
    if (do_stroke_collisions)
        max_sw = (50 + ppm * 64 * 10) / 100; /* 10% of an em for CJK */
    else
        max_sw = (50 + ppm * 64 * 22) / 100; /* 22% of an em otherwise */
    /* "Impact.ttf" is ~21% */

    /* vertical pieces and strokes -- don't do if font is italic */
    /* note: oblique does not matter ... just fonts drawn as italic */
    if (!is_italic || ref_lines[RTGA_l_RSB] == 0)
    {
        plist = (PLIST*)(gs->globalGS->rtgah_data.RTGAHStrokesPlist);
        plistMax = (PLIST*)(gs->globalGS->rtgah_data.RTGAHEdges);

#ifdef RTGAH_DEBUG
        FS_PRINTF(("x direction\n"));
#endif  /* RTGAH_DEBUG */

        if (script == SCRIPT_CJK)
        {
            n_pieces = find_pieces_CJK('x', pieces, elementPtr, &plist, plistMax);
            n_strokes = find_strokes_CJK('x', n_pieces, pieces, strokes, elementPtr, max_sw, &plist, plistMax);
        }
        else
        {
            n_pieces = find_pieces('x', pieces, elementPtr, &plist, plistMax);
            n_strokes = find_strokes('x', n_pieces, pieces, strokes, elementPtr, max_sw, &plist, plistMax);
        }

        /* this is executed only if we have not done the "l" yet */
        /* (which is assumed to be the first hinted character retrieved). */
        if (ref_lines[RTGA_l_RSB] == 0)
        {
            F26DOT6 c, w, a, b, L, R, S, lsb, rsb;
            FS_USHORT *ep;
            FS_USHORT nc, np;
            int index;

            nc = elementPtr->nc;
            ep = elementPtr->ep;
            np = 1 + ep[nc - 1];

            if (n_strokes == 0)
            {
                index = 0;
                strokes[0].width = 0;
                strokes[0].raw_center = 0;
            }
            else if (n_strokes == 1)
                index = 0;
            else
            {
                int jj;
                int minwidth = strokes[0].width;
                index = 0;

                for (jj = 1; jj < n_strokes; jj++)
                {
                    if (minwidth > strokes[jj].width)
                    {
                        minwidth = strokes[jj].width;
                        index = jj;
                    }
                }
            }


            c = strokes[index].raw_center;
            w = strokes[index].width / 2;
            a = elementPtr->x[np];      /* left side bearing point */
            b = elementPtr->x[1 + np];  /* right side bearing point */
            L = (c - w) - a;            /* white space on left */
            R = b - (c + w);            /* white space on right */
            S = ROUND_26_6(L + R);      /* round total space */
            lsb =  FLOOR_26_6(S / 2);   /* split evenly */
            rsb = S - lsb;              /* with excess to rsb */
            if (rsb < 64)
                rsb = 64;               /* at least one pixel */

            ref_lines[RTGA_l_LSB] = lsb;
            ref_lines[RTGA_l_RSB] = rsb;

            /* we don't want to try and autohint the font if */
            /*   too many points in the 'l'  ... 50 is a guess */
            /*   too many strokes in the 'l' ... 2 is a guess */
            /* */
            if (np > 50 && n_strokes > 2)
            {
                gs->globalGS->rtgah_data.rtgah_suitable = RTGAH_NOPE;
                SYS_MEMCPY(ref_lines, latin_rl, 6); /* restore Latin ref lines */
                return;
            }
        }

        /* if only autohinting in y (lite version), then skip x autohinting */
        if (!(gs->globalGS->rtgah_data.stateflags & FLAGS_AUTOHINT_YONLY_ON))
        {

            if (do_latin_hints)
            {
                int unicode = gs->globalGS->rtgah_data.unicode;

                latin_side_bearings(n_strokes, strokes, ref_lines, elementPtr);

                if (unicode == 0x69 || unicode == 0x6a)
                {
                    int i;
                    int top = 0;
                    F26DOT6 mindx = 300000;
                    int min = -1;
                    F26DOT6 ci;
                    /* fixup for 'i' and 'j' */
                    /* if two vertical strokes have (almost) the same raw-center, */
                    /* make sure the adjusted center's are equal */
                    for (i = 1; i < n_strokes; i++)
                    {
                        if (strokes[top].max < strokes[i].max)
                            top = i;
                    }

                    for (i = 0; i < n_strokes; i++)
                    {
                        F26DOT6 dx = strokes[top].raw_center - strokes[i].raw_center;
                        if (i == top)
                            continue;

                        if (ABS(dx) < mindx && ABS(dx) < 16)
                        {
                            mindx = ABS(dx);
                            min = i;
                        }
                    }
                    if (min != -1)
                    {
                        ci = simple_alignment(strokes[min].raw_center, strokes[min].width);
                        strokes[min].adj_center = strokes[top].adj_center = ci;
                    }
                }
            }


            if (do_stroke_collisions)
                no_collisions('x', n_strokes, strokes, elementPtr);

            adjust_outline('x', script, n_strokes, strokes, elementPtr, gs);
            if (script != SCRIPT_CJK)
            {
                gs->opCode = 1; /* 'x' */
                fnt_IUP(gs);
            }

#if defined(RTGAH_DEBUG) && (RTGAH_DEBUG & 128)
            dump_element("after iup(x)", elementPtr);
#endif
        }
    }

    /* horizontal strokes */
#ifdef RTGAH_DEBUG
    FS_PRINTF(("y direction\n"));
#endif  /* RTGAH_DEBUG */

    /* now the list of free PLIST objects can be reused for horizontal strokes */
    plist = (PLIST*)(gs->globalGS->rtgah_data.RTGAHStrokesPlist);
    plistMax = (PLIST*)(gs->globalGS->rtgah_data.RTGAHEdges);

    if (script == SCRIPT_CJK)
    {
        n_pieces = find_pieces_CJK('y', pieces, elementPtr, &plist, plistMax);
        n_strokes = find_strokes_CJK('y', n_pieces, pieces, strokes, elementPtr, max_sw, &plist, plistMax);
    }
    else
    {
        n_pieces = find_pieces('y', pieces, elementPtr, &plist, plistMax);
        n_strokes = find_strokes('y', n_pieces, pieces, strokes, elementPtr, max_sw, &plist, plistMax);
    }

    if (n_strokes)
    {
        /* return position of grid aligned top/bottom stroke edges */
        STROKE *s = strokes;
        F26DOT6 adj;

        adj = simple_alignment(s->raw_center, s->width);
        gs->globalGS->rtgah_data.lo = adj - s->width / 2;
        s += n_strokes - 1;
        adj = simple_alignment(s->raw_center, s->width);
        gs->globalGS->rtgah_data.hi = adj + s->width / 2;
    }

    /* different reference line combinations */
    switch (script)
    {
    case SCRIPT_CJK:
        rl4[0] = ref_lines[RTGA_CJK_PTOP];
        rl4[1] = ref_lines[RTGA_CJK_TOP];
        rl4[2] = ref_lines[RTGA_CJK_BOT];
        rl4[3] = ref_lines[RTGA_CJK_PBOT];
        snap_top_bottom_4(rl4, n_strokes, strokes, elementPtr, ppm << 6);
        break;

    case SCRIPT_HANGUL:
        rl4[0] = ref_lines[RTGA_HANGUL_PTOP];
        rl4[1] = ref_lines[RTGA_HANGUL_TOP];
        rl4[2] = ref_lines[RTGA_HANGUL_BOT];
        rl4[3] = ref_lines[RTGA_HANGUL_PBOT];
        snap_top_bottom_4(rl4, n_strokes, strokes, elementPtr, ppm << 6);
        break;

    case SCRIPT_DEVANAGARI:
        snap_top_bottom(ref_lines[RTGA_DEVANAGARI_TOP], n_strokes, strokes);
        break;

    case SCRIPT_BENGALI:
        snap_top_bottom(ref_lines[RTGA_BENGALI_TOP], n_strokes, strokes);
        break;

    case SCRIPT_GURMUHKI:
        snap_top_bottom(ref_lines[RTGA_GURMUKHI_TOP], n_strokes, strokes);
        break;

    case SCRIPT_GUJARATI:
        rl4[0] = ref_lines[RTGA_GUJARATI_CAP_ROUND];
        rl4[1] = ref_lines[RTGA_GUJARATI_CAP_SQUARE];
        rl4[2] = ref_lines[RTGA_GUJARATI_BASE_SQUARE];
        rl4[3] = ref_lines[RTGA_GUJARATI_BASE_ROUND];
        Indic_ref_lines_4(rl4, n_strokes, strokes);
        break;

    case SCRIPT_TELUGU:
        rl4[0] = ref_lines[RTGA_TELUGU_TOP];
        rl4[1] = ref_lines[RTGA_TELUGU_BOT];
        Telugu_ref_lines(rl4, n_strokes, strokes);
        break;

    case SCRIPT_KANNADA:
        /* Jelle liked the results better without refline processing */
        /* but we might want to try again ... interpolation has changed */

        /* rl4[0] = ref_lines[RTGA_KANNADA_CAP_ROUND]; */
        /* rl4[1] = ref_lines[RTGA_KANNADA_CAP_SQUARE]; */
        /* rl4[2] = ref_lines[RTGA_KANNADA_BASE_SQUARE]; */
        /* rl4[3] = ref_lines[RTGA_KANNADA_BASE_ROUND]; */
        /* Indic_ref_lines_4(rl4,n_strokes,strokes); */
        break;

    case SCRIPT_TAMIL:
        rl4[0] = ref_lines[RTGA_TAMIL_CAP_ROUND];
        rl4[1] = ref_lines[RTGA_TAMIL_CAP_SQUARE];
        rl4[2] = ref_lines[RTGA_TAMIL_BASE_SQUARE];
        rl4[3] = ref_lines[RTGA_TAMIL_BASE_ROUND];
        Indic_ref_lines_4(rl4, n_strokes, strokes);
        break;

    case SCRIPT_MALAYALAM:
        rl4[0] = ref_lines[RTGA_MALAYALAM_CAP_ROUND];
        rl4[1] = ref_lines[RTGA_MALAYALAM_CAP_SQUARE];
        rl4[2] = ref_lines[RTGA_MALAYALAM_BASE_SQUARE];
        rl4[3] = ref_lines[RTGA_MALAYALAM_BASE_ROUND];
        Indic_ref_lines_4(rl4, n_strokes, strokes);
        break;

    case SCRIPT_ORIYA:
        rl4[0] = ref_lines[RTGA_ORIYA_CAP_ROUND];
        rl4[1] = ref_lines[RTGA_ORIYA_CAP_SQUARE];
        rl4[2] = ref_lines[RTGA_ORIYA_BASE_SQUARE];
        rl4[3] = ref_lines[RTGA_ORIYA_BASE_ROUND];
        Indic_ref_lines_4(rl4, n_strokes, strokes);
        break;

    case SCRIPT_SINHALA:
        rl4[0] = ref_lines[RTGA_SINHALA_CAP_ROUND];
        rl4[1] = ref_lines[RTGA_SINHALA_CAP_SQUARE];
        rl4[2] = ref_lines[RTGA_SINHALA_BASE_SQUARE];
        rl4[3] = ref_lines[RTGA_SINHALA_BASE_ROUND];
        Indic_ref_lines_4(rl4, n_strokes, strokes);
        break;

    case SCRIPT_THAI:
        rl4[0] = ref_lines[RTGA_SINHALA_CAP_ROUND];
        rl4[1] = ref_lines[RTGA_SINHALA_CAP_SQUARE];
        rl4[2] = ref_lines[RTGA_SINHALA_BASE_SQUARE];
        rl4[3] = ref_lines[RTGA_SINHALA_BASE_ROUND];
        Indic_ref_lines_4(rl4, n_strokes, strokes);
        break; /* it's not Indic ... but the logic is correct. */

    case SCRIPT_LATIN:
    case SCRIPT_GREEK:
    case SCRIPT_CYRILLIC:
        n_strokes = latin_ref_lines(n_strokes, strokes, ref_lines, elementPtr, &plist, plistMax);
        n_strokes = latin_accents(ref_lines, n_strokes, strokes, elementPtr, &plist, plistMax, gs);
        break;

    case SCRIPT_KHMER:
        rl4[0] = ref_lines[RTGA_KHMER_CAP_ROUND];
        rl4[1] = rl4[0]; /* is no cap_square per se ... use cap_round here too */
        rl4[2] = ref_lines[RTGA_KHMER_BASE_SQUARE];
        rl4[3] = ref_lines[RTGA_KHMER_BASE_ROUND];
        Indic_ref_lines_4(rl4, n_strokes, strokes);
        break; /* it's not Indic ... but the logic is correct. */

    case SCRIPT_ARABIC:
        rl4[0] = ref_lines[RTGAH_ARABIC_BOT];
        Arabic_ref_lines(rl4, n_strokes, strokes);
        break;

    default:
        break;
    }

    if (do_stroke_collisions)
        no_collisions('y', n_strokes, strokes, elementPtr);

    adjust_outline('y', script, n_strokes, strokes, elementPtr, gs);
    if (script != SCRIPT_CJK)
    {
        gs->opCode = 0; /* 'y' */
        fnt_IUP(gs);
        if (gs->globalGS->rtgah_data.fnt_type == CFF_TYPE)
            adjust_outline_cff('y', elementPtr);
    }


    /* final pass for arabic and hebrew vowels */
    if (do_multi_dot)
        multi_dot(elementPtr);

    SYS_MEMCPY(ref_lines, latin_rl, 6 * sizeof(F26DOT6)); /* restore Latin ref lines */

}

/********************************************************************************/
#endif  /* FS_HINTS */


